#include "append.h"
#include "smtp_protocol.h"

#include <backend/yreflection_types.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/repository.h>
#include <yplatform/find.h>

#include <boost/lexical_cast.hpp>
#include <boost/regex.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <boost/uuid/random_generator.hpp>

namespace yimap { namespace backend {

namespace {

// legacy data definition from macs_ora still used in production

#define _IMF_SET_BIT(n) (1 << n)
//! bit flags from the is_mixed field
enum is_mixed_flags
{
    IMF_has_attachment = _IMF_SET_BIT(0),
    IMF_virus = _IMF_SET_BIT(1),
    IMF_spam = _IMF_SET_BIT(2),
    IMF_delivery = _IMF_SET_BIT(3),
    IMF_postmaster = _IMF_SET_BIT(4),
    IMF_recent = _IMF_SET_BIT(5),
    IMF_draft = _IMF_SET_BIT(6),
    IMF_deleted = _IMF_SET_BIT(7),
    IMF_rcpt_only = _IMF_SET_BIT(8),
    IMF_forwarded = _IMF_SET_BIT(9),
    IMF_answered = _IMF_SET_BIT(10),
    IMF_seen = _IMF_SET_BIT(11),
    IMF_no_body = _IMF_SET_BIT(12),
    IMF_no_answer = _IMF_SET_BIT(14),
    IMF_has_user_labels = _IMF_SET_BIT(15)
};
#undef _IMF_SET_BIT

}

//-----------------------------------------------------------------------------
// Email preprocessing

AppendEmail AppendEmail::create(const std::string& email)
{
    boost::regex yateamRegexp("^(.*)@(yandex-team.\\w+)$");
    boost::smatch match;
    if (boost::regex_match(email, match, yateamRegexp))
    {
        return AppendEmail(match[1] + "@mail." + match[2]);
    }
    else
    {
        return AppendEmail(email);
    }
}

//-----------------------------------------------------------------------------
// X-Yandex-Hint creating

AppendXYandexHint AppendXYandexHint::create(
    const string& fid,
    const string& date,
    const StringList& systemFlags,
    const StringList& userFlags)
{
    auto mixed = getIsMixed(systemFlags, userFlags);

    std::string userFlagsStr = "";
    for (StringList::const_iterator iter = userFlags.begin(); iter != userFlags.end(); iter++)
    {
        userFlagsStr += "imaplabel=" + *iter + "\n";
    }

    string noImapXYandexHintValue = "received_date=" + date + "\n" +
        "mixed=" + boost::lexical_cast<std::string>(mixed) + "\n" + userFlagsStr +
        "skip_loop_prevention=1\n" + "fid=" + fid + "\nnotify=0\nfilters=0";

    string XYandexHintValue = noImapXYandexHintValue + "\nimap=1";
    string XYandexHint;
    XYandexHint += "X-Yandex-Hint: ";
    XYandexHint += yplatform::base64_encode(XYandexHintValue.begin(), XYandexHintValue.end());
    XYandexHint += "\n";
    return AppendXYandexHint(XYandexHint);
}

size_t AppendXYandexHint::getIsMixed(const StringList& systemFlags, const StringList& userFlags)
{
    unsigned short isMixed = 0;
    for (auto& flag : systemFlags)
    {
        if (yplatform::iequals(flag, "deleted")) isMixed |= IMF_deleted;
        if (yplatform::iequals(flag, "recent")) isMixed |= IMF_recent;
        if (yplatform::iequals(flag, "answered")) isMixed |= IMF_answered;
        if (yplatform::iequals(flag, "draft")) isMixed |= IMF_draft;
        if (yplatform::iequals(flag, "seen")) isMixed |= IMF_seen;
    }

    for (auto& flag : userFlags)
    {
        if (yplatform::iequals("$forwarded", flag)) isMixed |= IMF_forwarded;
    }
    return isMixed;
}

//-----------------------------------------------------------------------------
// Message preprocessing

AppendMessage::Ptr AppendMessage::create(AppendRequest& request)
{
    auto hint = AppendXYandexHint::create(
        request.folder.fid, request.date, request.systemFlags, request.userFlags);
    auto uniq = getYandexUniq(request.messageBody);

    std::string message;
    message.reserve(request.messageBody.size() + uniq.size() + hint.size());
    message += hint;
    message += uniq;
    message += request.messageBody;

    AppendMessage::Ptr result(new AppendMessage{ std::move(message) });
    return result;
}

std::string AppendMessage::getYandexUniq(const string& body)
{
    size_t xuniqPos = body.find("X-Yandex-Uniq:");
    size_t headerEnd = body.find("\r\n\r\n");
    if (xuniqPos != string::npos && xuniqPos < headerEnd) return "";

    boost::uuids::random_generator gen;
    boost::uuids::uuid uuid = gen();
    ostringstream uniqStr;
    uniqStr << "X-Yandex-Uniq: " << uuid << "\r\n";
    return uniqStr.str();
}

//-----------------------------------------------------------------------------
// Url creating

AppendUrl AppendUrl::create(const AppendRequest& request)
{
    auto email = AppendEmail::create(request.email);

    std::stringstream url;

    url << "append?"
        << "src_email=" << email << "&"
        << "fid=" << request.folder.fid << "&";

    if (!request.userFlags.empty())
    {
        std::string uFlags =
            "user_flags=" + boost::algorithm::join(request.userFlags, "&user_flags=");
        url << uFlags << "&";
    }

    if (!request.systemFlags.empty())
    {
        std::string sFlags =
            "system_flags=" + boost::algorithm::join(request.systemFlags, "&system_flags=");
        url << sFlags << "&";
    }

    url << "date=" << request.date;

    return AppendUrl(url.str());
}

//-----------------------------------------------------------------------------
// Smtp sent logic

AppendResultFuture Append::append(AppendRequest&& request_)
{
    request = std::move(request_);

    bool useHttp = appendOpt.shouldUseSmtpgate(imap_ctx_->uniq_id());

    if (useHttp)
    {
        logger.logEvent() << "append byHttp to '" << request.folder.name
                          << "':" << request.folder.fid << " using:" << appendOpt.httpHost << " "
                          << imap_ctx_->uniq_id();

        auto host = appendOpt.httpHost + ":" + std::to_string(appendOpt.httpPort);
        auto url = AppendUrl::create(request);
        auto message = request.messageBody;
        sendByHttp(host, url, std::move(message));
    }
    else
    {
        logger.logEvent() << "append bySmtp to '" << request.folder.name
                          << "':" << request.folder.fid << " using:" << appendOpt.smtpHost << " "
                          << imap_ctx_->uniq_id();

        auto email = AppendEmail::create(request.email);
        auto message = AppendMessage::create(request);
        sendBySmtp(email, std::move(message));
    }

    return promise;
}

void Append::sendBySmtp(const AppendEmail& email, AppendMessage::Ptr&& message)
{
    mysmtp::SmtpClient smtp(logger, mysmtp::SmtpClient::lmtp_mode);
    mysmtp::RecipientList to;
    to.push_back(email);

    size_t retry_count = 0u;
    int ret_code = 0;
    while (retry_count < appendOpt.smtpRetryCount)
    {
        ret_code = smtp.connect(
            appendOpt.smtpHost, appendOpt.smtpPort, appendOpt.smtpTimeout, appendOpt.smtpIpv6);
        if (ret_code == 0)
        {
            ret_code = smtp.sendMail(std::string(""), to, message->data, imap_ctx_->uniq_id());
            if (ret_code == 0)
            {
                break;
            }
        }
        logger.logError() << std::string(" Append bySmtp failed. Retrying bySmtp. '")
                          << request.folder.name << std::string("':") << request.folder.fid;
        ++retry_count;
    }
    if (ret_code != 0)
    {
        smtp = mysmtp::SmtpClient(logger, mysmtp::SmtpClient::lmtp_mode);
        ret_code = smtp.connect(
            appendOpt.smtpHost, appendOpt.smtpPort, appendOpt.smtpTimeout, appendOpt.smtpIpv6);
        if (ret_code == 0)
        {
            ret_code = smtp.sendMail(std::string(""), to, message->data, imap_ctx_->uniq_id());
        }
    }

    if (ret_code == 0)
    {
        promise.set(AppendResult{ smtp.getSavedUid(), smtp.getSavedMid() });
    }
    else
    {
        promise.set_exception(AppendSendError("Append bySmtp failed.", request.folder));
    }
}

void Append::sendByHttp(const std::string& host, const AppendUrl& urlPath, std::string&& message)
{
    std::string url = host + std::string("/") + urlPath;
    auto request = yhttp::request::POST(url, std::move(message));

    auto httpService = yplatform::find<yhttp::call>("http_client");
    auto self = shared_from(this);

    auto cb = std::bind(&Append::httpCallBack, self, std::placeholders::_1, std::placeholders::_2);

    httpService->async_run(context, request, cb);
}

void Append::httpCallBack(const Error& err, Response res)
{
    std::stringstream error;

    if (err)
    {
        auto msg = err.message();
        error << std::string(" Append byHttp failed. ") + msg;
        return retryHttp(error.str());
    }

    if (res.status != 200)
    {
        error << std::string(" Append byHttp failed. ") + res.body.substr(0, 128);
        return retryHttp(error.str());
    }

    try
    {
        auto result = yamail::data::deserialization::fromJson<AppendResult>(res.body);
        promise.set(result);
    }
    catch (const std::exception& e)
    {
        error << std::string(" Append byHttp failed. ") + e.what();
        return retryHttp(error.str());
    }
    catch (...)
    {
        error << std::string(" Append byHttp failed. Unknown error.");
        return retryHttp(error.str());
    }
}

void Append::retryHttp(const std::string& error)
{
    if (httpRetryCount < appendOpt.httpRetryCount)
    {
        logger.logError() << error << " Retrying byHttp. '" << request.folder.name
                          << std::string("':") << request.folder.fid;
        ++httpRetryCount;
        auto host = appendOpt.httpHost + ":" + std::to_string(appendOpt.httpPort);
        auto url = AppendUrl::create(request);
        auto message = request.messageBody;
        sendByHttp(host, url, std::move(message));
    }
    else if (appendOpt.httpRetrySmtp)
    {
        logger.logError() << error << " Retrying bySmtp. '" << request.folder.name
                          << std::string("':") << request.folder.fid;
        auto email = AppendEmail::create(request.email);
        auto message = AppendMessage::create(request);
        sendBySmtp(email, std::move(message));
    }
    else
    {
        promise.set_exception(AppendSendError(error, request.folder));
    }
}

} // namespace backend
} // namespace yimap
