#pragma once

#include <common/types.h>
#include <common/imap_context.h>

#include <backend/backend.h> // StringList
#include <backend/cache_worker.h>
#include <backend/backend_types.h>

#include <ymod_httpclient/call.h>

#include <common/folder.h>
#include <boost/shared_ptr.hpp>
#include <sstream>
#include <memory>

namespace yimap { namespace backend {

//-----------------------------------------------------------------------------
// Append request

struct AppendRequest
{
    std::string email;
    std::string date;
    StringList systemFlags;
    StringList userFlags;
    std::string messageBody;
    DBFolderId folder;
};

//-----------------------------------------------------------------------------
// Append request

struct AppendResult
{
    std::string uid;
    std::string mid;
};

using AppendResultFuture = Future<AppendResult>;
using AppendResultPromise = Promise<AppendResult>;

//-----------------------------------------------------------------------------
// Email preprocessor

// In case of ya-team we should use <username>@mail.yandex-team.ru instead of
// <username>@yandex-team.ru [DARIA-27900]
struct AppendEmail : public std::string
{
    static AppendEmail create(const std::string& email);

protected:
    AppendEmail(const std::string& email) : std::string(email)
    {
    }
};

//-----------------------------------------------------------------------------
// X-Yandex-Hint helper required to strong typing

struct AppendXYandexHint : public std::string
{
    static size_t getIsMixed(const StringList& systemFlags, const StringList& userFlags);
    static AppendXYandexHint create(
        const std::string& fid,
        const std::string& date,
        const StringList& systemFlags,
        const StringList& userFlags);

protected:
    AppendXYandexHint(const std::string& data) : std::string(data)
    {
    }
};

struct AppendMessage
{
    const std::string data;

    using Ptr = std::unique_ptr<AppendMessage>;
    static AppendMessage::Ptr create(AppendRequest& request);

protected:
    static std::string getYandexUniq(const std::string& body);
};

//-----------------------------------------------------------------------------
// Url creating

struct AppendUrl : public std::string
{
    static AppendUrl create(const AppendRequest& request);

protected:
    AppendUrl(const std::string& data) : std::string(data)
    {
    }
};

//-----------------------------------------------------------------------------
// X-Yandex-Hint helper required to strong typing

class Append
    : public CacheWorker
    , public std::enable_shared_from_this<Append>
{
public:
    Append(ImapContextPtr context)
        : CacheWorker(context->settings, context)
        , context(context)
        , logger(context->sessionLogger)
        , appendOpt(settings_->appendSettings)
    {
    }

    using Error = boost::system::error_code;
    using Response = yhttp::response;

    AppendResultFuture append(AppendRequest&& request);
    void httpCallBack(const Error& err, Response res);

protected:
    void sendBySmtp(const AppendEmail& email, AppendMessage::Ptr&& message);
    void sendByHttp(const std::string& host, const AppendUrl& url, std::string&& message);
    void retryHttp(const std::string& error);

    ImapContextPtr context;
    Logger& logger;
    const AppendSettings& appendOpt;
    AppendRequest request;
    AppendResultPromise promise;
    size_t httpRetryCount = 0u;
};

struct AppendSendError : public runtime_error
{
    explicit AppendSendError(const std::string message, const DBFolderId& folder)
        : runtime_error(message + " '" + folder.name + "':" + folder.fid)
    {
    }
};

} // namespace backend
} // namespace yimap
