#include <backend/append/smtp_protocol.h>
#include <boost/lexical_cast.hpp>
#include <vector>
//#include <errno.h>

#include <iostream>

namespace yimap { namespace mysmtp {

const int SMTP_OK = 250;

std::string SmtpClient::error_msg(int error_code)
{
    switch (error_code)
    {
    case 0:
        return "No error";
    case -1:
        return "Operation failed";
    case -2:
        return "Operation timed out";
    default:
    {
        if (error_code > 0) return last_error;
        else
            return "Unknown error";
    }
    }
}

int SmtpClient::connect(const std::string& hostname, int port, int timeout, bool ipv6)
{
    conn_info.ipv6 = ipv6;
    auto res = connect_(hostname, port, timeout);
    if (res != 0)
    {
        logger->logError() << "Connect to " << hostname << ":" << port << " failed.  At "
                           << remote_smtp_host << " Error: " << error_msg(res);
        return res;
    }
    return 0;
}

int SmtpClient::sendMail(
    const std::string& from,
    const RecipientList& to,
    const std::string& data,
    const std::string& envId)
{
    int res;
    RecipientList::const_iterator to_failed;
    res = sendEnvelope(from, to, to_failed, envId);
    if (res != 0)
    {
        std::string s;
        if (to_failed != to.end()) s = ". Failed recipient: <" + to_failed->email_ + ">";
        logger->logError() << "Send envelope failed. At " << remote_smtp_host
                           << "Error: " << error_msg(res) << s;
        return SC_FAIL_ENVELOPE;
    }
    res = sendMailData(data);
    if (res != 0)
    {
        logger->logError() << "Send mail data failed. At " << remote_smtp_host
                           << " Error: " << error_msg(res);
        return SC_FAIL_DATA;
    }
    return 0;
}

int SmtpClient::connect_(const std::string& hostname, int port, int timeout)
{
    int err = makeConnection(hostname, port, timeout);
    if (err != 0)
    {
        return err;
    }
    SmtpResult res = readResult();
    if (res.code != 220) return res.code;

    remote_smtp_host = last_error;
    size_t start = remote_smtp_host.find(" ");
    if (start != std::string::npos)
    {
        remote_smtp_host = remote_smtp_host.erase(0, start + 1);
        size_t end = remote_smtp_host.find(" ");
        remote_smtp_host = remote_smtp_host.substr(0, end);
    }
    res = query((lmtp ? "LHLO " : "EHLO ") + hostname + "\r\n");
    if (res.code != SMTP_OK) return res.code;
    params._8bitmime = res.text.find("8BITMIME") != std::string::npos;
    params.pipelining = res.text.find("PIPELINING") != std::string::npos;
    params.dsn = res.text.find("DSN") != std::string::npos;

    return 0;
}

int SmtpClient::sendEnvelope(
    const std::string& from,
    const RecipientList& to,
    RecipientList::const_iterator& to_failed,
    const std::string& envId)
{
    to_failed = to.end();
    std::string mailFrom = "MAIL FROM: <" + from + ">";
    if (!envId.empty()) mailFrom += " ENVID=" + envId;
    mailFrom += "\r\n";
    SmtpResult res = query(mailFrom);
    if (res.code != SMTP_OK) return res.code;

    for (RecipientList::const_iterator it = to.begin(); it != to.end(); ++it)
    {
        std::string rcpt = "RCPT TO: <" + it->email_ + ">";
        if (it->requestDsn_) rcpt += " NOTIFY=SUCCESS,FAILURE";
        rcpt += "\r\n";
        res = query(rcpt);
        if (res.code != SMTP_OK)
        {
            to_failed = it;
            logger->logError() << "Send rcpt failed. At " << remote_smtp_host << ". Rcpt  "
                               << it->email_ << " . Error: " << error_msg(res.code);
            return res.code;
        }
    }

    return 0;
}

int SmtpClient::sendMailData(const std::string& data)
{
    SmtpResult res = query("DATA\r\n");
    if (res.code != 354) return res.code;

    std::string esc_data = data;
    size_t pos = 0;
    while ((pos = esc_data.find("\r\n.\r\n", pos)) != std::string::npos)
    {
        esc_data = esc_data.insert(pos + 2, ".");
    }
    pos = 0;
    while ((pos = esc_data.find("\n.\n", pos)) != std::string::npos)
    {
        esc_data = esc_data.insert(pos + 1, ".");
    }
    if (!esc_data.empty() && esc_data[esc_data.length() - 1] == '\n') esc_data += ".\r\n";
    else
        esc_data += "\r\n.\r\n";

    res = query(esc_data);
    if (res.code != SMTP_OK) return res.code;
    else
    {
        std::vector<std::string::size_type> spaces;
        spaces.push_back(0);
        std::string::size_type newspace = 0;
        while (newspace != std::string::npos)
        {
            newspace = last_error.find(" ", newspace + 1);
            spaces.push_back(newspace + 1);
        }
        enum
        {
            CODE,
            RESPCODE,
            RESPSTATUS,
            TRANSACTIONID,
            MID,
            UID
        };
        std::string::size_type END = spaces.size() - 1;
        saved_mid =
            (END > MID) ? last_error.substr(spaces[MID], (spaces[MID + 1] - spaces[MID])) : "";
        for (std::string::size_type i = 0; i < saved_mid.size(); i++)
            if (saved_mid[i] < '0' || saved_mid[i] > '9') saved_mid.erase(i--, 1);
        saved_uid =
            (END > UID) ? last_error.substr(spaces[UID], (spaces[UID + 1] - spaces[UID])) : "";
        if (saved_uid.substr(0, 4) == "none") saved_uid = "none";
        else
            for (std::string::size_type i = 0; i < saved_uid.size(); i++)
                if (saved_uid[i] < '0' || saved_uid[i] > '9') saved_uid.erase(i--, 1);
    }

    return 0;
}

void SmtpClient::close()
{
    query("QUIT\r\n");
    close_connection();
}

int SmtpClient::makeConnection(const std::string& host, int port, int timeout)
{
    return sock_wrap_connect(host.c_str(), port, timeout, conn_info);
}

void SmtpClient::close_connection()
{
    sock_wrap_close(conn_info);
}

SmtpResult SmtpClient::query(const std::string& data)
{
    if (sendQuery(data) != 0)
    {
        SmtpResult res;
        res.code = -1;
        return res;
    }
    return readResult();
}

int SmtpClient::sendQuery(const std::string& data)
{
    int res = sock_wrap_write(data.c_str(), data.size(), conn_info);
    if (res < 0 || data.size() != size_t(res)) return -1;
    return 0;
}

SmtpResult SmtpClient::readResult()
{
    SmtpResult res;
    while (1)
    {
        std::string line;
        int res1;
        if ((res1 = readLine(line)) != 0)
        {
            res.code = res1;
            return res;
        }
        if (line.length() >= 4 && line[3] == ' ')
        {
            int code = boost::lexical_cast<int>(line.substr(0, 3));
            if (code)
            {
                res.code = code;
                last_error = line;
                return res;
            }
        }
        res.text += line;
    }
}

int SmtpClient::readLine(std::string& line)
{
    size_t pos;
    while ((pos = buffer.find('\n')) == std::string::npos)
    {
        int res;
        if ((res = readBuffer()) != 0) return res;
    }
    line = buffer.substr(0, pos + 1);
    buffer.erase(0, pos + 1);
    return 0;
}

int SmtpClient::readBuffer()
{
    char buf[2049];
    int len = sock_wrap_read(buf, sizeof(buf) - 1, conn_info);
    if (len <= 0)
    {
        return len;
    }
    buf[len] = 0;
    buffer += buf;
    return 0;
}

}}
