#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <netdb.h>
#include <errno.h>
#include <unistd.h>
#include <poll.h>
#include <stdio.h>

#include <backend/append/sock_wrap.h>

#ifndef INADDR_NONE
#define INADDR_NONE -1
#endif

namespace yimap {

ssize_t sock_wrap_connect(
    const char* hostname,
    const int port,
    const int timeout,
    TConnectionInfo& info)
{
    info.socket = -1;
    info.timeout = 0;

    auto sock = socket(info.ipv6 ? PF_INET6 : PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock <= 0) return -1;

    ssize_t connectResult = 0;
    if (info.ipv6)
    {
        struct hostent* server = gethostbyname2(hostname, AF_INET6);
        if (server == nullptr)
        {
            close(sock);
            return -1;
        }
        struct sockaddr_in6 sin6;
        memset(&sin6, 0, sizeof(sin6));
        sin6.sin6_family = AF_INET6;
        memmove((char*)&sin6.sin6_addr.s6_addr, (char*)server->h_addr, server->h_length);
        sin6.sin6_port = htons(static_cast<uint16_t>(port));

        if (fcntl(sock, F_SETFL, fcntl(sock, F_GETFL, 0) | O_NONBLOCK) == -1)
        {
            close(sock);
            return -1;
        }

        connectResult = connect(sock, (struct sockaddr*)&sin6, sizeof(sin6));
    }
    else
    {
        struct sockaddr_in sin4;
        memset(&sin4, 0, sizeof(sin4));
        sin4.sin_family = AF_INET;
        sin4.sin_port = htons(port);

        sin4.sin_addr.s_addr = inet_addr(hostname); // IP adress ??

        if (sin4.sin_addr.s_addr == INADDR_NONE)
        {
            struct hostent hst;
            struct hostent* start = NULL;
            char buffer[2048];
            int err;

            gethostbyname_r(hostname, &hst, buffer, sizeof(buffer), &start, &err);

            if (start == NULL)
            {
                close(sock);
                return -1;
            }
            sin4.sin_addr.s_addr = ((struct in_addr*)start->h_addr)->s_addr;
        }

        if (fcntl(sock, F_SETFL, fcntl(sock, F_GETFL, 0) | O_NONBLOCK) == -1)
        {
            close(sock);
            return -1;
        }
        connectResult = connect(sock, (struct sockaddr*)&sin4, sizeof(sin4));
    }

    if (connectResult == 0)
    {
        fcntl(sock, F_SETFL, fcntl(sock, F_GETFL, 0) & ~O_NONBLOCK);
        info.socket = sock;
        info.timeout = timeout;
        return 0;
    }

    struct pollfd fd;
    fd.fd = sock;
    fd.events = POLLOUT;
    int pres;

    if ((pres = poll(&fd, 1, timeout * 1000)) == -1)
    {
        close(sock);
        return -1;
    }

    if (pres > 0)
    {
        fcntl(sock, F_SETFL, fcntl(sock, F_GETFL, 0) & ~O_NONBLOCK);
        info.socket = sock;
        info.timeout = timeout;
        return 0;
    }

    // errno = ETIMEDOUT;

    close(sock);

    return -2;
};

ssize_t sock_wrap_write(const char* buf, ssize_t len, TConnectionInfo& info)
{
    struct pollfd fd;
    fd.fd = info.socket;
    fd.events = POLLOUT;
    ssize_t pres = 1;
    ssize_t flen = len;

    while (len > 0 && (pres > 0 || (pres < 0 && (errno == EINTR || errno == EAGAIN))))
    {
        if ((pres = poll(&fd, 1, info.timeout * 1000)) == -1) return -1;

        if (pres > 0)
        {
            pres = write(info.socket, buf, len);
            if (pres > 0)
            {
                buf += pres;
                len -= pres;
            }
        }
        else
            return -2;
    }

    if (pres > 0) return flen;
    else
        return pres;
}

ssize_t sock_wrap_read(char* buf, ssize_t len, TConnectionInfo& info)
{
    struct pollfd fd;
    fd.fd = info.socket;
    fd.events = POLLIN;
    ssize_t pres;

    if ((pres = poll(&fd, 1, info.timeout * 1000)) == -1) return -1;

    ssize_t ret;

    if (pres > 0)
    {
        do
        {
            errno = 0;
            ret = read(info.socket, buf, len);
        } while ((ret <= 0) && (errno == EINTR || errno == EAGAIN));
    }
    else
    {
        // errno = ETIMEDOUT;
        return -2;
    }

    if ((ret == 0) && (errno == 0)) // buffer empty and no err
    {
        // errno = ETIMEDOUT;
        return -2;
    }

    return ret;
};

void sock_wrap_close(TConnectionInfo& info)
{
    if (info.socket != -1)
    {
        close(info.socket);
        info.socket = -1;
    }
};

}
