#include <backend/auth_ymodbb/auth_backend_ymodbb.h>

#include <common/crc.h>
#include <ymod_blackbox/client.h>
#include <ymod_blackbox/options.h>
#include <ymod_blackbox/error.h>
#include <yplatform/find.h>

#include <boost/property_tree/xml_parser.hpp>

namespace yimap { namespace backend {

const ymod_blackbox::db_fields_list AuthBackendYmodbb::blackbox_db_fields = {
    ymod_blackbox::db_field("subscription.login.2"),
    ymod_blackbox::db_field("subscription.login.8"),
    ymod_blackbox::db_field("subscription.login_rule.8"),
    ymod_blackbox::db_field("subscription.suid.2"),
    ymod_blackbox::db_field("hosts.db_id.2"),
    ymod_blackbox::db_field("userinfo.lang.uid")
};

const string AuthBackendYmodbb::enable_app_password_attribute = { "107" };

AuthBackendYmodbb::~AuthBackendYmodbb()
{
}

AuthBackend::LoginRes AuthBackendYmodbb::asyncLogin(string login, string pass)
{
    user = login;
    password = pass;
    passwordCRC = crc(pass.data(), pass.size());
    ++triesCount;

    auto authModule = yplatform::find<ymod_blackbox::client>("blackbox");
    authModule->async_login(
        login,
        auth_settings.service,
        pass,
        { context->sessionInfo.remoteAddress, context->sessionInfo.remotePort },
        std::bind(
            &AuthBackendYmodbb::loginCallback,
            shared_from(this),
            std::placeholders::_1,
            std::placeholders::_2),
        ymod_blackbox::options_list(),
        blackbox_db_fields,
        { enable_app_password_attribute });

    return promise;
}

void AuthBackendYmodbb::loginCallback(
    const ymod_blackbox::error& err,
    const ymod_blackbox::login_response& resp)
{
    if (err)
    {
        logFailGetResult(err.code.message() + ", " + err.ext_reason);

        auto maxTries = context->settings->authRetry + 1;
        if (triesCount < maxTries)
        {
            retryAuth(std::bind(&AuthBackendYmodbb::asyncLogin, shared_from(this), user, password));
        }
        else
        {
            AuthResult authResult;
            authResult.loginFail = true;
            authResult.tmpFail = true;
            authResult.errorStr = err.code.message();
            promise.set(authResult);
        }
        return;
    }

    bool loginValid = resp.status == ymod_blackbox::login_response::valid;
    if (!loginValid)
    {
        loginFailed(resp, "blackbox login not success");
        return;
    }

    bool userBlocked =
        context->settings->enableBBBlock && !checkLoginRule(resp["subscription.login_rule.8"]);
    if (userBlocked)
    {
        loginFailed(resp, "user blocked in blackbox");
        return;
    }

    bool emptySuid = resp["subscription.suid.2"].empty();
    if (emptySuid)
    {
        loginFailed(resp, "blackbox return empty user data");
    }
    else
    {
        loginSuccess(resp);
    }
}

AuthBackend::LoginRes AuthBackendYmodbb::asyncLoginOAuth(string token)
{
    oauth_token = token;
    ++triesCount;

    auto authModule = yplatform::find<ymod_blackbox::client>("blackbox");
    authModule->async_oauth(
        token,
        { context->sessionInfo.remoteAddress, context->sessionInfo.remotePort },
        std::bind(
            &AuthBackendYmodbb::oauthCallback,
            shared_from(this),
            std::placeholders::_1,
            std::placeholders::_2),
        ymod_blackbox::options_list(),
        blackbox_db_fields);

    return promise;
}

void AuthBackendYmodbb::oauthCallback(
    const ymod_blackbox::error& err,
    const ymod_blackbox::session_id_response& resp)
{
    if (err)
    {
        logFailGetResult(err.code.message() + ", " + err.ext_reason);

        auto maxTries = context->settings->authRetry + 1;
        if (triesCount < maxTries)
        {
            retryAuth(
                std::bind(&AuthBackendYmodbb::asyncLoginOAuth, shared_from(this), oauth_token));
        }
        else
        {
            AuthResult authResult;
            authResult.loginFail = true;
            authResult.tmpFail = true;
            authResult.errorStr = err.code.message();
            promise.set(authResult);
        }
        return;
    }

    bool loginValid = resp.status == ymod_blackbox::login_response::valid;
    if (!loginValid)
    {
        loginFailed(resp, "blackbox login not success");
        return;
    }

    bool userBlocked =
        context->settings->enableBBBlock && !checkLoginRule(resp["subscription.login_rule.8"]);
    if (userBlocked)
    {
        loginFailed(resp, "user blocked in blackbox");
        return;
    }

    bool emptySuid = resp["subscription.suid.2"].empty();
    if (emptySuid)
    {
        loginFailed(resp, "blackbox return empty user data");
        return;
    }

    bool scopeCorrect = checkOAuthScope(resp.raw_response());
    if (!scopeCorrect)
    {
        loginFailed(resp, "invalid oauth scope");
    }
    else
    {
        loginSuccess(resp);
    }
}

bool AuthBackendYmodbb::checkKarma(const string& karma)
{
    if (!context->settings->checkKarma) return true;

    return !(karma == "100");
}

bool AuthBackendYmodbb::checkLoginRule(const string& rule)
{
    try
    {
        int login_rule_val = boost::lexical_cast<int>(rule);
        return !(login_rule_val & 0x04); // don't ask me why
    }
    catch (const std::exception& e)
    {
        return false;
    }
}

bool AuthBackendYmodbb::checkOAuthScope(const string& raw_resp)
{
    auto oauthResp = getRespPtree(raw_resp);
    string scopeString = oauthResp.get<string>("doc.OAuth.scope", "");

    set<string> scopes;
    boost::split(scopes, scopeString, boost::is_any_of(" "));

    set<string> scopeIntersection;
    set_intersection(
        scopes.begin(),
        scopes.end(),
        context->settings->oauthScope.begin(),
        context->settings->oauthScope.end(),
        std::inserter(scopeIntersection, scopeIntersection.begin()));

    return !scopeIntersection.empty();
}

void AuthBackendYmodbb::logFailGetResult(const string& exception)
{
    context->sessionLogger.logError()
        << "auth_backend:" << user << "' pass_crc=" << passwordCRC
        << " addr=" << context->sessionInfo.remoteAddress << ':' << context->sessionInfo.remotePort
        << " result=FAIL module=blackbox"
        << " reason='ymod_blackbox error (" << exception << ")'";
}

void AuthBackendYmodbb::logFailAuth(
    const string& message,
    const string& karma,
    const string& ban_time)
{
    context->sessionLogger.logError()
        << "auth_backend: user='" << user << "' pass_crc=" << passwordCRC
        << " addr=" << context->sessionInfo.remoteAddress << ':' << context->sessionInfo.remotePort
        << " result=FAIL module=blackbox"
        << " message=" << message << " karma=" << karma << " ban_time=" << ban_time;
}

boost::property_tree::ptree AuthBackendYmodbb::getRespPtree(const string& raw_resp)
{
    stringstream xmlstream;
    xmlstream << raw_resp;
    boost::property_tree::ptree respTree;
    boost::property_tree::read_xml(xmlstream, respTree);
    return respTree;
}

}} // namespace
