#ifndef AUTHBACKENDYMODBB_H_
#define AUTHBACKENDYMODBB_H_

#include <backend/backend.h>
#include <common/settings.h>
#include <common/crc.h>
#include <ymod_blackbox/client.h>

namespace yimap { namespace backend {
namespace detail {

inline bool hasDotInLogin(const string& email)
{
    string::size_type p = email.find('.');
    if (p == string::npos) return false;

    string::size_type d = email.find('@');
    return (d == string::npos || p < d);
}

}

class AuthBackendYmodbb
    : public AuthBackend
    , public std::enable_shared_from_this<AuthBackendYmodbb>
{
public:
    AuthBackendYmodbb(ImapContextPtr context, const AuthSettings& auth_settings)
        : context(context), auth_settings(auth_settings)
    {
    }
    virtual ~AuthBackendYmodbb();

    virtual LoginRes asyncLogin(string login, string pass);
    virtual LoginRes asyncLoginOAuth(string token);

private:
    void loginCallback(const ymod_blackbox::error& err, const ymod_blackbox::login_response& resp);
    void oauthCallback(
        const ymod_blackbox::error& err,
        const ymod_blackbox::session_id_response& resp);

    template <typename AuthFunction>
    void retryAuth(AuthFunction&& function)
    {
        auto retryTimeout = Milliseconds(context->settings->authRetryTimeout);
        retryTimer = createTimer(context, retryTimeout, function);
    }

    bool checkOAuthScope(const string& resp);
    bool checkKarma(const string& karma);
    bool checkLoginRule(const string& rule);

    void logFailGetResult(const string& exception);
    void logFailAuth(const string& message, const string& karma, const string& ban_time);

    template <typename Response>
    void loginFailed(const Response& resp, const string& reason)
    {
        ymod_blackbox::karma_info_option karma_info;
        resp >> karma_info;

        ymod_blackbox::uid_option uid_info;
        resp >> uid_info;

        AuthResult authResult;
        authResult.loginFail = true;
        authResult.tmpFail = false;
        authResult.karmaFail = !checkKarma(karma_info.karma);
        authResult.uid = uid_info.uid;
        authResult.suid = resp["subscription.suid.2"];
        authResult.language = resp["userinfo.lang.uid"];
        authResult.errorStr = reason + ": " + resp.message();

        logFailAuth(resp.message(), karma_info.karma, karma_info.ban_time);
        promise.set(authResult);
    }

    template <typename Response>
    void loginSuccess(const Response& resp)
    {
        ymod_blackbox::karma_info_option karma_info;
        resp >> karma_info;

        ymod_blackbox::uid_option uid_info;
        resp >> uid_info;

        ymod_blackbox::connection_id_option conn_id_info;
        resp >> conn_id_info;

        AuthResult authResult;
        authResult.karmaFail = !checkKarma(karma_info.karma);
        authResult.loginFail = false;
        authResult.uid = uid_info.uid;
        authResult.suid = resp["subscription.suid.2"];
        authResult.language = resp["userinfo.lang.uid"];
        authResult.storage = resp["hosts.db_id.2"];
        authResult.connectionId = conn_id_info.value;
        authResult.login = resp["subscription.login.2"];

        authResult.email = resp["subscription.login.8"];
        if (!detail::hasDotInLogin(authResult.email))
        {
            authResult.email = resp["subscription.login.2"];
        }

        if (authResult.email.find('@') == string::npos)
        {
            if (context->settings->defaultDomain.size())
            {
                authResult.email.append("@");
                authResult.email.append(context->settings->defaultDomain);
            }
        }

        if (authResult.email.find('@') == string::npos)
        {
            std::size_t pos;
            if ((pos = user.find('@')) != string::npos)
            {
                authResult.email.append(user, pos, user.length());
            }
            else
            {
                auto defaultDomain = auth_settings.domain;
                if (defaultDomain.size() > 0 && defaultDomain[0] != '@')
                {
                    authResult.email.append("@");
                }
                authResult.email.append(defaultDomain);
            }
        }

        auto attrs = bb::Attributes(&resp.native_response());
        authResult.authByApplicationPassword = attrs.get(enable_app_password_attribute) == "1";

        auto respPtree = getRespPtree(resp.raw_response());
        authResult.domainId = respPtree.get("doc.uid.<xmlattr>.domid", "");
        promise.set(authResult);
    }

    boost::property_tree::ptree getRespPtree(const string& raw_resp);

private:
    static const ymod_blackbox::db_fields_list blackbox_db_fields;
    static const string enable_app_password_attribute;

    ImapContextPtr context;
    AuthSettings auth_settings;

    string user;
    string password;
    CRC32Value passwordCRC;
    string oauth_token;

    LoginPromise promise;
    std::size_t triesCount = 0;
    TimerPtr retryTimer;
};

}} // namespace

#endif /* AUTHBACKENDYMODBB_H_ */
