#pragma once

#include "handlers.h"
#include <common/message_list_diff.h>
#include <common/folder.h>
#include <common/folder_list.h>
#include <common/settings.h>
#include <common/uid_map.h>
#include <common/imap_context.h>
#include <backend/events.h>
#include <backend/backend_types.h>

#include <user_journal/parameters/message.h>
#include <user_journal/parameters/parameter_id.h>

namespace yimap {

namespace util {
class seq_range;
}

using StringList = std::vector<std::string>;

typedef Future<void> future_void_t;
typedef std::function<void(future_void_t)> CacheResultHook;

struct BodyStructure;

namespace backend {

class MetaBackend
{
public:
    virtual ~MetaBackend() = default;

    virtual Future<FolderListPtr> getFolders() = 0;
    virtual Future<FolderPtr> getFolder(const DBFolderId& folderId) = 0;
    virtual Future<FolderInfo> getFolderInfo(const DBFolderId& folderId) = 0;

    virtual Future<UidMapPtr> loadMessages(
        FolderRef mailbox,
        const seq_range& range,
        bool partial = true) = 0;
    virtual Future<UidMapPtr> loadMessagesChunk(FolderRef mailbox, const seq_range& range) = 0;
    virtual Future<UidMapPtr> loadMessagesDetails(FolderRef mailbox, const seq_range& range) = 0;
    virtual Future<UidMapPtr> loadMessagesToDelete(FolderRef mailbox, const seq_range& range) = 0;
    virtual Future<UidMapPtr> getMessagesByMessageId(
        FolderRef mailbox,
        const string& messageId) = 0;
    virtual Future<BodyMetadataByMid> loadBodyMetadata(const SmidList& mids) = 0;

    virtual Future<MailboxDiffPtr> statusUpdate(FolderRef mailbox) = 0;

    virtual Future<MailboxDiffPtr> updateFlags(
        FolderRef mailbox,
        UidMapPtr uids,
        const Flags& del_flags,
        const Flags& add_flags) = 0;

    virtual Future<void> expunge(FolderRef mailbox, UidMapPtr uids) = 0;

    virtual Future<UidSequence> copyMessages(
        UidMap& uids,
        const FolderInfo& from,
        const FolderInfo& to) = 0;
    virtual Future<UidSequence> moveMessages(
        UidMap& uids,
        const FolderInfo& from,
        const FolderInfo& to) = 0;

    virtual Future<void> createFolder(const string& folderName) = 0;
    virtual Future<void> deleteFolder(const DBFolderId& folderId) = 0;
    virtual Future<void> renameFolder(const DBFolderId& folderId, const string& dstName) = 0;
    virtual Future<void> subscribe(const DBFolderId& folderId, bool on) = 0;

    virtual Future<void> dropFreshCounter() = 0;
    virtual void journalAuth() = 0;

    virtual Future<string> regenerateImapId(const std::string& mid) = 0;

    using Authorization = user_journal::parameters::
        OperationParameters<user_journal::Target::account, user_journal::Operation::authorization>;
};

typedef std::shared_ptr<MetaBackend> MetaBackendPtr;

struct AuthResult : UserData
{
    bool loginFail = true;
    bool karmaFail = false;
    bool tmpFail = false;

    std::string errorStr;
};

class AuthBackend
{
public:
    typedef Future<AuthResult> LoginRes;
    typedef Promise<AuthResult> LoginPromise;

    virtual ~AuthBackend(){};

    virtual LoginRes asyncLogin(string login, string pass) = 0;
    virtual LoginRes asyncLoginOAuth(string token) = 0;
};

typedef std::shared_ptr<AuthBackend> AuthBackendPtr;

class UserSettingsBackend
{
public:
    virtual ~UserSettingsBackend() = default;
    virtual Future<UserSettings> loadSettings() = 0;
};

typedef std::shared_ptr<UserSettingsBackend> UserSettingsBackendPtr;

class NotificationsBackend
{
public:
    virtual ~NotificationsBackend() = default;
    virtual void subscribe(WeakEventHandlerPtr handler) = 0;
};

typedef std::shared_ptr<NotificationsBackend> NotificationsBackendPtr;

class SearchBackend
{
public:
    using SearchCallback = std::function<void(ErrorCode, std::set<string>)>;

    virtual ~SearchBackend() = default;

    virtual void search(
        const string& searchRequest,
        const string& originalRequest,
        const string& fid,
        const SearchCallback& cb) = 0;
};

typedef std::shared_ptr<SearchBackend> SearchBackendPtr;

class MbodyBackend
{
public:
    typedef std::function<void(const string& err, StringPtr msg)> Handler;

    virtual ~MbodyBackend() = default;

    virtual void loadMessage(
        const std::string& stid,
        const std::string& part,
        const macs::MimeParts& mimeParts,
        const Handler& handler) = 0;

    virtual void loadHeader(
        const std::string& stid,
        const std::string& part,
        const macs::MimeParts& mimeParts,
        const Handler& handler) = 0;

    virtual void loadBody(
        const std::string& stid,
        const std::string& part,
        const macs::MimeParts& mimeParts,
        const Handler& handler) = 0;

    virtual Future<std::shared_ptr<BodyStructure>> loadBodyStructure(
        const string& stid,
        const macs::MimeParts& mimeParts,
        const std::vector<string>& rawHeadersList) = 0;
};

typedef std::shared_ptr<MbodyBackend> MbodyBackendPtr;

struct AppendRequest;
struct AppendResult;

struct AppendBackend
{
    virtual ~AppendBackend() = default;

    virtual Future<AppendResult> append(AppendRequest&& request) = 0;
};

typedef std::shared_ptr<AppendBackend> AppendBackendPtr;

struct Bundle
{
    AuthBackendPtr authBackend;
    MetaBackendPtr metaBackend;
    MbodyBackendPtr mbodyBackend;
    UserSettingsBackendPtr settingsBackend;
    UserJournalPtr journal;
    NotificationsBackendPtr notificationsBackend;
    SearchBackendPtr search;
    AppendBackendPtr append;
};
using BundlePtr = std::shared_ptr<Bundle>;

FolderPtr CreatePgMappedFolder(
    const FolderInfo& folderInfo,
    const UidMapData& uidMapData,
    bool isShared);

} // namespace backend

struct FolderInfoError : public runtime_error
{
    explicit FolderInfoError(const string& folderName, const string& message)
        : runtime_error(
              string("(FolderInfoError) Failed to load folder info '") + folderName +
              "': " + message)
    {
    }
};

struct UidMapError : public runtime_error
{
    explicit UidMapError(const string& folderName, const string& message)
        : runtime_error(
              string("(UidMapError) Failed to load uid map for '") + folderName + "': " + message)
    {
    }
};

struct LoadMessagesError : public runtime_error
{
    explicit LoadMessagesError(const string& folderName, const string& message)
        : runtime_error(
              string("(LoadMessagesError) Failed to load messages from '") + folderName +
              "': " + message)
    {
    }
};

struct StatusUpdateError : public runtime_error
{
    explicit StatusUpdateError(const string& folderName, const string& message)
        : runtime_error(string("Failed to load changes for '") + folderName + "': " + message)
    {
    }
};

struct ExpungeError : public runtime_error
{
    explicit ExpungeError(const string& folderName, const string& message)
        : runtime_error(string("Expunge failed for '") + folderName + "': " + message)
    {
    }
};

struct CopyError : public runtime_error
{
    explicit CopyError(const string& from, const string& to, const string& message)
        : runtime_error(
              string("Can't copy messages from '") + from + "' to '" + to + "':" + message)
    {
    }
};

struct MoveError : public runtime_error
{
    explicit MoveError(const string& from, const string& to, const string& message)
        : runtime_error(
              string("Can't copy messages from '") + from + "' to '" + to + "':" + message)
    {
    }
};

struct FolderListError : public runtime_error
{
    explicit FolderListError(const string& reason)
        : runtime_error(string("Failed to load folders list :") + reason)
    {
    }
};

struct FolderNameTooLargeError : public runtime_error
{
    explicit FolderNameTooLargeError(const string& folderName, const string& what)
        : runtime_error(what), folderName(folderName)
    {
    }

    const std::string folderName;
};

struct LoadSettingsError : public runtime_error
{
    explicit LoadSettingsError(const string& user, const string& error)
        : runtime_error("failed to load settings user="s + user + " error=" + error)
    {
    }
};

} // namespace yimap
