#include "envelope.h"
#include <backend/literal_out.h>
#include <yplatform/util.h>
#include <parser/rfc822/rfc822.h>
#include <boost/algorithm/string.hpp>

namespace yimap { namespace backend {

using namespace yimap::rfc822::rfc2822;

bool create_address_list(EnvelopeData::addr_list_t& addr_list, const FieldValuePtr& fv_ptr)
{
    {
        const AddressFieldValue* ptr = dynamic_cast<const AddressFieldValue*>(fv_ptr.get());

        if (ptr)
        {
            addr_list.clear();
            addr_list.push_back(AddrStructure());

            if (!ptr->addr.name.empty())
                addr_list.back().display_name = msg_trim_str(ptr->addr.name);

            if (!ptr->addr.local.empty())
                addr_list.back().local_name = msg_trim_str(ptr->addr.local);

            if (!ptr->addr.domain.empty())
                addr_list.back().domain_name = msg_trim_str(ptr->addr.domain);

            return true;
        }
    }

    {
        const AddressListFieldValue* ptr = dynamic_cast<const AddressListFieldValue*>(fv_ptr.get());

        if (ptr)
        {
            addr_list.clear();
            for (AddressType const& addr : ptr->addrs)
            {
                addr_list.push_back(AddrStructure());

                if (!addr.name.empty()) addr_list.back().display_name = msg_trim_str(addr.name);

                if (!addr.local.empty()) addr_list.back().local_name = msg_trim_str(addr.local);

                if (!addr.domain.empty()) addr_list.back().domain_name = msg_trim_str(addr.domain);
            }
            return true;
        }
    }

    return false;
}

struct EnvelopeActions : public rfc822::MessageReactor
{
    EnvelopeData& env;
    inline explicit EnvelopeActions(EnvelopeData& envelope) : env(envelope)
    {
    }
    void on_field_data(const rfc822::FieldData& fd) override;
};

void EnvelopeActions::on_field_data(const rfc822::FieldData& fd)
{
    string fieldName = boost::to_lower_copy(fd.name);
    if (fieldName == "date")
    {
        env.date = msg_trim_str(fd.value->raw);
        return;
    }

    if (fieldName == "subject")
    {
        env.subject = msg_trim_str(fd.value->raw);
        return;
    }

    if (fieldName == "in-reply-to")
    {
        env.in_reply_to = msg_trim_str(fd.value->raw);
        return;
    }

    if (fieldName == "message-id")
    {
        env.message_id = msg_trim_str(fd.value->raw);
        return;
    }

    if (fieldName == "from")
    {
        create_address_list(env.from, fd.value);

        if (env.sender.empty()) env.sender = env.from;
        if (env.reply_to.empty()) env.reply_to = env.from;
        return;
    }

    if (fieldName == "sender")
    {
        // do not change prev value with empty string
        if (!msg_trim_str(fd.value->raw).empty() || env.sender.empty())
            create_address_list(env.sender, fd.value);
        return;
    }

    if (fieldName == "reply_to")
    {
        // do not change prev value with empty string
        if (!msg_trim_str(fd.value->raw).empty() || env.reply_to.empty())
            create_address_list(env.reply_to, fd.value);
        return;
    }

    if (fieldName == "to")
    {
        create_address_list(env.to, fd.value);
        return;
    }

    if (fieldName == "cc")
    {
        create_address_list(env.cc, fd.value);
        return;
    }

    if (fieldName == "bcc")
    {
        create_address_list(env.bcc, fd.value);
        return;
    }
}

bool parse_envelope(
    const string& data,
    EnvelopeData& envelope,
    const std::vector<std::string>& rawHeadersList)
{
    EnvelopeActions actions(envelope);
    return rfc822::parseMessage(data, actions, rawHeadersList);
}

}}
