#pragma once

#include <backend/literal_out.h>
#include <common/types.h>

namespace yimap { namespace backend {

typedef boost::optional<string> StringOpt;

struct AddrStructure
{
    StringOpt display_name;
    StringOpt source_route;
    StringOpt local_name;
    StringOpt domain_name;
};

struct EnvelopeData
{
    typedef std::list<AddrStructure> addr_list_t;

    StringOpt date;
    StringOpt subject;
    addr_list_t from;     // mailbox-list
    addr_list_t sender;   // mailbox
    addr_list_t reply_to; // address-list
    addr_list_t to;       // address-list
    addr_list_t cc;       // address-list
    addr_list_t bcc;      // address-list
    StringOpt in_reply_to;
    StringOpt message_id;

    template <typename StreamT>
    void send(StreamT& stream) const
    {
        stream << "(";
        send(stream, date);
        stream << ' ';
        send(stream, subject);
        stream << ' ';
        send(stream, from);
        stream << ' ';
        send(stream, sender);
        stream << ' ';
        send(stream, reply_to);
        stream << ' ';
        send(stream, to);
        stream << ' ';
        send(stream, cc);
        stream << ' ';
        send(stream, bcc);
        stream << ' ';
        send(stream, in_reply_to);
        stream << ' ';
        send(stream, message_id);
        stream << ')';
    }

    template <typename StreamT>
    void send(StreamT& stream, const string& data) const
    {
        boost::iterator_range<string::const_iterator> rng(data.begin(), data.end());
        literalOut(stream, data.empty() ? 0 : &rng);
    }

    template <typename StreamT>
    void send(StreamT& stream, const StringOpt& data) const
    {
        if (!data)
        {
            literalOut(stream, static_cast<boost::iterator_range<string::const_iterator>*>(0));
            return;
        }
        boost::iterator_range<string::const_iterator> rng(data->begin(), data->end());
        literalOut(stream, &rng);
    }

    template <typename StreamT>
    void send(StreamT& stream, const addr_list_t& data) const
    {
        if (data.empty())
        {
            stream << "NIL";
            return;
        }

        stream << '(';

        bool need_space = false;
        for (typename addr_list_t::const_iterator i = data.begin(); i != data.end(); ++i)
        {
            if (need_space) stream << ' ';
            else
                need_space = true;

            stream << '(';
            send(stream, i->display_name);
            stream << ' ';
            send(stream, i->source_route);
            stream << ' ';
            send(stream, i->local_name);
            stream << ' ';
            send(stream, i->domain_name);
            stream << ')';
        }

        stream << ')';
    }
};

bool parse_envelope(
    const string& data,
    EnvelopeData& envelope,
    const std::vector<std::string>& rawHeadersList);

} // namespace backend
} // namespace yimap
