#pragma once

#include <common/imap_context.h>
#include <common/types.h>

#include <yplatform/find.h>
#include <boost/property_tree/json_parser.hpp>

namespace yimap {

class ImapEvent
{
public:
    virtual ~ImapEvent() = default;

    virtual string name() const = 0;
    virtual string action() const = 0;
    virtual string error() const = 0;

    virtual bool checkConditions(ImapContextPtr context) const = 0;
};
typedef boost::shared_ptr<ImapEvent> ImapEventPtr;

class ImapUnconditionalEvent : public ImapEvent
{
public:
    ImapUnconditionalEvent(const string& name, const string& error = "")
        : name_(name), errorMessage(error)
    {
    }
    virtual ~ImapUnconditionalEvent()
    {
    }

    virtual string name() const
    {
        return name_;
    };
    virtual string action() const
    {
        return "";
    };
    virtual string error() const
    {
        return errorMessage;
    };
    virtual bool checkConditions(ImapContextPtr) const
    {
        return true;
    };

private:
    string name_;
    string errorMessage;
};

class ImapTimedEvent : public ImapEvent
{
public:
    ImapTimedEvent(
        const string& name,
        const string& action,
        int64_t timestamp,
        const string& connectionId)
        : name_(name), action_(action), timestamp_(timestamp), connectionId_(connectionId)
    {
    }

    virtual ~ImapTimedEvent()
    {
    }

    virtual string name() const
    {
        return name_;
    };
    virtual string action() const
    {
        return action_;
    };
    virtual string error() const
    {
        return std::string();
    };
    virtual bool checkConditions(ImapContextPtr context) const
    {
        if (timestamp_ > context->userData.authTime)
        {
            if (connectionId_.empty()) return true;
            return connectionId_ == context->userData.connectionId;
        }
        return false;
    };

private:
    string name_;
    string action_;
    uint64_t timestamp_;
    string connectionId_;
};

}