#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/log.h>
#include <yplatform/module.h>
#include <yplatform/find.h>

#include <common/imap_context.h>

#include <macs/hooks.h>
#include <macs_pg/macs_pg.h>
#include <macs_pg/integration/pa_profiler.h>

#include <backend/backend_service.h>
#include <backend/sharpei_resolver.hpp>

#include <boost/algorithm/string/replace.hpp>
#include <pa/async.h>
#include <macs/user_journal.h>
#include <user_journal/parameters/parameter_id.h>
#include <user_journal/logging.h>
#include <user_journal/service.h>
#include <user_journal/service_factory.h>

using namespace user_journal;

namespace yimap { namespace backend {

namespace {

template <typename Duration>
auto toFloatSeconds(const Duration& d)
{
    return durationCast<FloatSeconds>(d).count();
}

}

class SharpeiHttp;

class UserJournalLogger : public logging::Log
{
public:
    void warning(const std::string&, const std::string&, const std::string&, int) override
    {
    }
    void error(const std::string&, const std::string&, const std::string&, int) override
    {
    }
    void notice(const std::string&, const std::string&, const std::string&) override
    {
    }
    void debug(const std::string&, const std::string&, const std::string&) override
    {
    }
};

class MacsLogger : public macs::pg::logging::Log
{
public:
    MacsLogger(const ShortSessionInfo& sessionInfo) : sessionInfo(sessionInfo)
    {
    }

    void warning(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs warning from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void error(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs error from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void notice(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(error) << sessionInfo.logPrefix() << "Macs notice from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }
    void debug(const macs::pg::logging::Method& method, const macs::pg::logging::Message& message)
        override
    {
        L_(debug) << sessionInfo.logPrefix() << "Macs debug from " << method << ": "
                  << boost::replace_all_copy(message, "\n", "\\n");
    }

protected:
    ShortSessionInfo sessionInfo;
};

//-----------------------------------------------------------------------------
// Backend service to manage DB and backend requests

class BackendServiceImpl
    : public yplatform::module
    , public BackendService
{
public:
    BackendServiceImpl(boost::asio::io_service& io, const Ptree& xml);
    virtual ~BackendServiceImpl();

    virtual void fini();
    virtual void start()
    {
    }
    virtual void stop()
    {
    }

    Ptree get_stats() const override;

    void writeTskv(const std::string& record);

    UserJournalPtr createJournal(
        const std::string& uid,
        const std::string& clientIp,
        const std::string& mdb,
        const std::string& suid,
        const std::string& service) override;
    PgServicePtr createPgService(
        const ShortSessionInfo& sessionInfo,
        bool replica,
        const std::string& service) override;

    ThreadPoolPtr getThreadPool() override
    {
        return threadPool;
    }

protected:
    void initLogger(const Ptree& xml);
    void initUserJournal(const Ptree& xml);
    void initPg(const Ptree& xml);
    void readSharpeiSettings(const Ptree& xml);
    user_journal::Journal journal(
        const std::string& uid,
        const std::string& clientIp,
        const std::string& service,
        const std::string& mdb,
        const std::string& suid);
    void initThreadPool(boost::asio::io_service& io, const Ptree& xml);

private:
    ::macs::pg::ConnectionPoolPtr pgPool_;
    ::macs::pg::QueryConf queryConf;
    ::macs::pg::SharpeiParams sharpeiParams;
    ::macs::pg::ServiceFactory::Milliseconds pqTransactionTimeout;
    bool pgProfilerEnabled = false;

    std::shared_ptr<user_journal::Service> userJournalService;
    boost::shared_ptr<yplatform::log::source> tskvLog;
    ThreadPoolPtr threadPool;
};

BackendServiceImpl::BackendServiceImpl(boost::asio::io_service& io, const Ptree& xml)
{
    L_(debug) << "BackendServiceImpl task instaniated";

    if (xml.get("pa.<xmlattr>.init", 0))
    {
        std::size_t max_size = xml.get("pa.<xmlattr>.max", 1000000);
        std::size_t dump_size = xml.get("pa.<xmlattr>.dump", 4096);
        std::string path = xml.get("pa.<xmlattr>.path", "");
        pa::async_profiler::init(max_size, dump_size, path);
    }

    initLogger(xml);
    initUserJournal(xml);
    initPg(xml);
    initThreadPool(io, xml.get_child("thread_pool"));
}

BackendServiceImpl::~BackendServiceImpl()
{
    L_(debug) << "BackendServiceImpl task destroyed";
}

void BackendServiceImpl::initLogger(const Ptree& xml)
{
    if (xml.get("write_tskv", 0))
    {
        tskvLog.reset(new yplatform::log::source(YGLOBAL_LOG_SERVICE, "user_journal_tskv"));
    }
}

void BackendServiceImpl::initUserJournal(const Ptree& xml)
{
    user_journal::ServiceFactory factory;

    factory.logger(boost::make_shared<UserJournalLogger>());
    factory.fileWriter(boost::bind(&BackendServiceImpl::writeTskv, this, _1));
    factory.tableName(xml.get("user_journal.table_name", "users_history"));
    factory.tskvFormat(xml.get("user_journal.tskv_format", "mail-user-journal-log"));

    userJournalService = factory.product();
}

void BackendServiceImpl::initPg(const Ptree& xml)
{
    auto ioService = yplatform::find_reactor(xml.get<std::string>("pg.reactor"))->io();
    pgPool_ = ::macs::pg::ConnectionPoolFactory()
                  .ioService(*ioService)
                  .connectTimeout(Milliseconds(xml.get("pg.connect_timeout", 100)))
                  .queueTimeout(Milliseconds(xml.get("pg.queue_timeout", 100)))
                  .queryTimeout(Milliseconds(xml.get("pg.query_timeout", 15000)))
                  .maxConnections(xml.get("pg.max_connections", 30))
                  .asyncResolve(xml.get("pg.async_resolve", true))
                  .ipv6Only(xml.get("pg.ipv6_only", true))
                  .dnsCacheTTL(Seconds(xml.get("pg.dns.cache_ttl", 120)))
                  .product();
    pqTransactionTimeout =
        ::macs::pg::ServiceFactory::Milliseconds(xml.get("pg.transaction_timeout", 30000));
    pgProfilerEnabled = xml.get("pg.log_pa", false);

    sharpeiParams = readSharpeiParams(xml);
    queryConf = ::macs::pg::readQueryConfFile(xml.get<std::string>("query_conf"));
}

void BackendServiceImpl::initThreadPool(boost::asio::io_service& io, const Ptree& xml)
{
    auto pool = yplatform::find_reactor(xml.get<std::string>("reactor"))->get_pool();
    auto rateController =
        ymod_ratecontroller::create_controller(io, pool->size(), xml.get("max_queue_size", 10000u));
    threadPool = std::make_shared<ThreadPool>(*pool->io(), rateController);
}

void BackendServiceImpl::writeTskv(const std::string& record)
{
    if (tskvLog)
    {
        auto& logger = *tskvLog;
        YLOG(logger, info) << record;
    }
}

void BackendServiceImpl::fini()
{
    pgPool_.reset();
}

Ptree BackendServiceImpl::get_stats() const
{
    auto pool_stats = pgg::statistic::getPoolStats(*pgPool_);
    Ptree pool_stats_node;
    for (auto& pair : pool_stats)
    {
        auto& conn_id = pair.first;
        auto& pool_stat = pair.second;

        Ptree stats_node;
        stats_node.put("free_connections", pool_stat.freeConnections);
        stats_node.put("busy_connections", pool_stat.busyConnections);
        stats_node.put("pending_connections", pool_stat.pendingConnections);
        stats_node.put("max_connections", pool_stat.maxConnections);
        stats_node.put("queue_size", pool_stat.queueSize);
        stats_node.put(
            "average_request_roundtrip", toFloatSeconds(pool_stat.averageRequestRoundtrip));
        stats_node.put(
            "average_request_db_latency", toFloatSeconds(pool_stat.averageRequestDbLatency));
        stats_node.put("average_wait_time", toFloatSeconds(pool_stat.averageWaitTime));
        stats_node.put("dropped_timeout", pool_stat.droppedTimedOut);
        stats_node.put("dropped_failed", pool_stat.droppedFailed);
        stats_node.put("dropped_busy", pool_stat.droppedBusy);
        stats_node.put("dropped_with_result", pool_stat.droppedWithResult);
        pool_stats_node.push_back(std::make_pair(conn_id, stats_node));
    }
    Ptree stats;
    stats.put_child("pool_stats", pool_stats_node);
    return stats;
}

//-----------------------------------------------------------------------------
// PG service

PgServicePtr BackendServiceImpl::createPgService(
    const ShortSessionInfo& sessionInfo,
    bool replica,
    const std::string& service)
{
    auto uidResolverFactory = macs::pg::createSharpeiUidResolverFactory(sharpeiParams);

    macs::pg::RequestInfo requestInfo;
    requestInfo.userIp = sessionInfo.clientIp;
    requestInfo.clientType = "imap";

    auto pgFactory = ::macs::pg::createSeviceFactory(pgPool_, uidResolverFactory);
    pgFactory->queryConf(queryConf);
    pgFactory->credentials(sharpeiParams.credentials);
    pgFactory->transactionTimeout(pqTransactionTimeout);
    if (pgProfilerEnabled)
    {
        pgFactory->profiler(macs::pg::integration::PaProfiler::create());
    }
    if (userJournalService)
    {
        pgFactory->userJournal(journal(
            sessionInfo.userData.uid,
            sessionInfo.clientIp,
            service,
            sessionInfo.userData.storage,
            ""));
    }
    pgFactory->logger(boost::make_shared<MacsLogger>(sessionInfo));
    pgFactory->autoQueryHandleStartegy(
        replica ? ::macs::pg::readReplicaThenMaster : ::macs::pg::readMasterThenReplica);
    pgFactory->requestInfo(std::move(requestInfo));

    return pgFactory->product(sessionInfo.userData.uid);
}

//-----------------------------------------------------------------------------
// User journal logging

class ImapUserJournal : public macs::UserJournal
{
public:
    using Base = macs::UserJournal;

    ImapUserJournal(Base::Journal journal, std::string mdb)
        : Base(std::move(journal)), mdb(std::move(mdb))
    {
    }

private:
    std::string mdb;

    void asyncGetShardName(OnStringReceive handler) const override
    {
        handler(mdb);
    }
};

UserJournalPtr BackendServiceImpl::createJournal(
    const std::string& uid,
    const std::string& clientIp,
    const std::string& mdb,
    const std::string& suid,
    const std::string& service)
{
    return std::make_shared<ImapUserJournal>(journal(uid, clientIp, service, mdb, suid), mdb);
}

user_journal::Journal BackendServiceImpl::journal(
    const std::string& uid,
    const std::string& clientIp,
    const std::string& service,
    const std::string& mdb,
    const std::string& suid)
{
    if (!userJournalService)
    {
        throw std::runtime_error("UserJournalService is't initialized");
    }

    namespace id = user_journal::parameters::id;

    return userJournalService->createJournal(
        uid, id::module(service), id::ip(clientIp), id::suid(suid));
}

} // namespace backend
} // namespace yimap

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yimap::backend::BackendServiceImpl)
