#pragma once

#include <common/types.h>

#include <yplatform/encoding/iterators/remove_nl.h>
#include <iostream>

#include <boost/optional.hpp>
#include <boost/algorithm/string.hpp>

namespace yimap { namespace backend {

// is white space
struct is_ws
{
    inline bool operator()(char const& x) const
    {
        return x == ' ' || x == '\t' || x == '\r' || x == '\n';
    }
};

namespace {
const is_ws is_ws_f = is_ws();
}

template <typename RangeT>
RangeT msg_trim(RangeT const& src)
{
    RangeT rng = boost::trim_copy_if(src, is_ws_f);

    // skip dquotes
    if (boost::starts_with(rng, "\"") && boost::ends_with(rng, "\""))
    {
        rng.advance_begin(1);
        if (boost::ends_with(rng, "\"")) rng.advance_end(-1);
    }

    return rng;
}

template <typename RangeT>
string msg_trim_str(RangeT const& src)
{
    RangeT rng = msg_trim(src);
    return string(rng.begin(), rng.end());
}

template <>
inline string msg_trim_str(const string& src)
{
    typedef boost::iterator_range<string::const_iterator> RangeT;
    RangeT rng = msg_trim(RangeT(src.begin(), src.end()));
    return string(rng.begin(), rng.end());
}

template <typename StreamT, typename RangeT>
void literalOut(StreamT& stream, RangeT const* t)
{
    using yplatform::iterators::remove_nl;

    if (!t)
    {
        stream << "NIL";
        return;
    }
    if (t->empty())
    {
        stream << "\"\"";
        return;
    }

    string ret("");

    int quoted = 0;
    bool literal = false;
    bool non_digits = false;

    typedef remove_nl<typename boost::range_iterator<RangeT>::type> remove_nl_iterator;
    typedef boost::iterator_range<remove_nl_iterator> remove_nl_range;

    remove_nl_range rng(t->begin(), t->end());

    for (typename remove_nl_range::reference x : rng)
    {
        if (x < '0' || x > '9') non_digits = true;
        if (x == ' ') ++quoted;
        if (x == '"' || x == '\\' || x == '\r' || x == '\n' || x == 0x00 || (x & 0x80) != 0)
        {
            literal = true;
            break;
        }
    }

    if (literal)
    {
        stream << '{' << rng.size() << "}\r\n" << rng;
        return;
    }

    // E.g. subjects may consist of digits only, and OE report parse error on unquoted subject
    if (1 || non_digits) ++quoted;
    if (quoted) stream << '"';
    stream << rng;
    if (quoted) stream << '"';

    // print in quoted form
}

template <typename StreamT, typename RangeT>
void literalOut(StreamT& stream, const boost::optional<RangeT>& t)
{
    literalOut(stream, t ? &*t : nullptr);
}

}}
