#pragma once

#include "body_structure_data.h"
#include <backend/literal_out.h>
#include <yplatform/log.h>
#include <yplatform/task_context.h>
#include <parser/rfc822/rfc822.h>
#include <list>

#include <boost/algorithm/string.hpp>

namespace yimap {

using backend::msg_trim_str;

struct BodyStructureActions : public rfc822::MessageReactor
{
    BodyStructure& structure;
    yplatform::task_context_ptr context;

    explicit BodyStructureActions(yplatform::task_context_ptr ctx, BodyStructure& s)
        : structure(s), context(ctx)
    {
    }

    virtual void on_field_data(const rfc822::rfc2822::FieldData& fd);
};

inline void BodyStructureActions::on_field_data(const rfc822::rfc2822::FieldData& fd)
{
    using namespace yimap::rfc822::rfc2822;

    string fieldName = boost::to_lower_copy(fd.name);
    if (fieldName == "content-type")
    {
        boost::shared_ptr<mime_content_type_field_value> ct =
            boost::dynamic_pointer_cast<mime_content_type_field_value>(fd.value);

        if (!ct)
        {
            TASK_LOG(context, warning) << "body structure: cannot parse content-type";
            structure.mime_type = "";
            structure.mime_subtype = "";
            structure.mime_params.clear();
            return;
        }

        if (structure.mime_type)
        {
            TASK_LOG(context, warning)
                << "body structure: several content-type headers in same message part";
        }

        structure.mime_type = msg_trim_str(ct->content_type.type);
        structure.mime_subtype = msg_trim_str(ct->content_type.subtype);

        structure.mime_params.clear();
        for (const mime_parameter& par : ct->params)
        {
            structure.mime_params.push_back(
                AttrStructure{ msg_trim_str(par.attr), msg_trim_str(par.value) });
        }

        return;
    }

    {
        StringOpt* val = 0;
        BodyStructure::attr_list_t* params = 0;

        if (fieldName == "content-disposition")
        {
            val = &structure.mime_disp;
            params = &structure.mime_disp_params;
        }
        else if (fieldName == "content-language")
        {
            val = &structure.mime_lang;
            params = &structure.mime_lang_params;
        }
        else if (fieldName == "content-location")
        {
            val = &structure.mime_location;
        }
        else if (fieldName == "content-id")
        {
            val = &structure.mime_id;
        }
        else if (fieldName == "content-description")
        {
            val = &structure.mime_description;
            if (fd.value && !fd.value->raw.empty()) *val = msg_trim_str(fd.value->raw);
            else
                *val = "";
            return;
        }
        else if (fieldName == "content-transfer-encoding")
        {
            val = &structure.mime_encoding;
        }
        else if (fieldName == "content-md5")
        {
            val = &structure.mime_md5;
            if (fd.value && !fd.value->raw.empty()) *val = msg_trim_str(fd.value->raw);
            else
                *val = "";
            return;
        }

        if (val)
        {
            boost::shared_ptr<mime_with_params_field_value> mp =
                boost::dynamic_pointer_cast<mime_with_params_field_value>(fd.value);

            if (!mp)
            {
                TASK_LOG(context, warning)
                    << "body structure: cannot parse mime header: " << fd.name;
                *val = "";
                if (params) params->clear();
                return;
            }

            if ((*val) || (params && !params->empty()))
            {
                TASK_LOG(context, warning) << "body_structure: dublicated mime header: " << fd.name;
                if (params) params->clear();
            }

            *val = msg_trim_str(mp->value);

            if (params)
            {
                for (const mime_parameter& par : mp->params)
                {
                    params->push_back(
                        AttrStructure{ msg_trim_str(par.attr), msg_trim_str(par.value) });
                }
            }
        }
    }
}

} // namespace yimap
