#pragma once

#include <common/types.h>
#include <backend/envelope.h>
#include <yplatform/util.h>

namespace yimap {

using backend::EnvelopeData;

typedef boost::optional<string> StringOpt;
template <typename StreamT>
inline void literal_out_str(StreamT& stream, const StringOpt& str)
{
    if (str)
    {
        boost::iterator_range<const char*> range(str->data(), str->data() + str->length());
        backend::literalOut(stream, &range);
    }
    else
    {
        backend::literalOut(stream, static_cast<const boost::iterator_range<const char*>*>(0));
    }
}

struct AttrStructure
{
    string attr;
    string value;
};

struct BodyStructure
{
    typedef std::list<AttrStructure> attr_list_t;
    typedef boost::optional<attr_list_t> opt_attr_list_t;
    typedef boost::optional<EnvelopeData> opt_envelope_data_t;

    StringOpt mime_type;
    StringOpt mime_subtype;

    attr_list_t mime_params;

    // content-disposition
    StringOpt mime_disp;
    attr_list_t mime_disp_params;

    // content-language
    StringOpt mime_lang;
    attr_list_t mime_lang_params;

    // content-location
    StringOpt mime_location;

    // content-id
    StringOpt mime_id;

    // content-description
    StringOpt mime_description;

    // content-encoding
    StringOpt mime_encoding;

    // content-md5
    StringOpt mime_md5;

    // envelope
    opt_envelope_data_t envelope;

    // HotFix rfcSize
    size_t rfc_size = 0;

    std::size_t body_size = 0;
    std::size_t body_lines = 0;

    std::list<BodyStructure> mime_parts;

    template <typename StreamT>
    void send(StreamT& stream, const StringOpt& data) const
    {
        literal_out_str(stream, data);
    }

    template <typename StreamT>
    void send(StreamT& stream, BodyStructure::attr_list_t const& list) const
    {
        if (list.empty())
        {
            stream << "NIL";
            return;
        }

        stream << '(';
        bool need_space = false;
        for (auto i = list.begin(); i != list.end(); ++i)
        {
            if (need_space) stream << ' ';
            else
                need_space = true;
            literal_out_str(stream, i->attr);
            stream << ' ';
            literal_out_str(stream, i->value);
        }
        stream << ')';
    }

    template <typename StreamT>
    void send(StreamT& stream, bool extended) const
    {
        stream << "(";

        bool multipart = false;
        if (mime_type && ::yplatform::util::iequals(*mime_type, "multipart"))
        {
            // multipart
            multipart = true;
            for (const auto& part : mime_parts)
            {
                part.send(stream, extended);
            }
        }
        else
        {
            if (mime_type) literal_out_str(stream, *mime_type);
            else
                stream << "\"TEXT\"";
        }

        stream << ' ';
        if (mime_subtype) literal_out_str(stream, mime_subtype);
        else
            stream << "\"PLAIN\"";

        if (multipart)
        {
            send_multipart(stream, extended);
        }
        else
        { // end part
            send_single(stream, extended);
        }

        stream << ')';
    }

    template <typename StreamT>
    void send_single(StreamT& stream, bool extended) const
    {
        stream << ' ';

        if (!mime_params.empty()) send(stream, mime_params);
        else
            stream << "NIL";

        stream << ' ';
        literal_out_str(stream, mime_id);

        stream << ' ';
        literal_out_str(stream, mime_description);

        stream << ' ';
        // http://support.microsoft.com/kb/975918/en-us
        // http://tools.ietf.org/html/rfc2045#section-6.1
        // According to RFC, content-transfer-encoding cannot be empty
        if (!mime_encoding || mime_encoding->empty()) stream << "\"7BIT\"";
        else
            literal_out_str(stream, mime_encoding);

        stream << ' ' << body_size;

        if (mime_type && mime_subtype && yplatform::util::iequals(*mime_type, "message") &&
            yplatform::util::iequals(*mime_subtype, "rfc822"))
        {
            stream << ' ';
            if (envelope) envelope->send(stream);
            else
                stream << "NIL";

            stream << ' ';
            for (BodyStructure const& part : mime_parts)
            {
                part.send(stream, extended);
                break;
            }

            stream << ' ' << body_lines;
        }
        else if (!mime_type || yplatform::util::iequals(*mime_type, "text"))
        {
            stream << ' ' << body_lines;
        }

        if (extended)
        {
            stream << ' ';
            literal_out_str(stream, mime_md5);
            stream << ' ';
            // disposition
            if (!mime_disp)
            {
                stream << "NIL";
            }
            else
            {
                stream << '(';
                literal_out_str(stream, mime_disp);
                stream << ' ';

                send(stream, mime_disp_params);
                stream << ')';
            }

            stream << ' ';
            if (!mime_lang)
            {
                stream << "NIL";
            }
            else
            {
                if (mime_lang_params.empty())
                {
                    literal_out_str(stream, mime_lang);
                }
                else
                {
                    stream << '(';
                    literal_out_str(stream, mime_lang);
                    stream << ' ';
                    send(stream, mime_lang_params);
                    stream << ')';
                }
            }

            stream << ' ';
            // location
            literal_out_str(stream, mime_location);
        }
    }

    template <typename StreamT>
    void send_multipart(StreamT& stream, bool extended) const
    {
        if (extended)
        {
            stream << ' ';

            if (!mime_params.empty()) send(stream, mime_params);
            else if (
                !mime_type || yplatform::util::iequals(*mime_type, "text") ||
                yplatform::util::iequals(*mime_type, "message"))
                stream << "(\"CHARSET\" \"X-UNKNOWN\")";

            stream << ' ';

            // disposition
            if (!mime_disp)
            {
                stream << "NIL";
            }
            else
            {
                stream << '(';
                literal_out_str(stream, mime_disp);
                stream << ' ';

                send(stream, mime_disp_params);
                stream << ')';
            }

            stream << ' ';

            // language
            if (!mime_lang)
            {
                stream << "NIL";
            }
            else
            {
                if (mime_lang_params.empty())
                {
                    literal_out_str(stream, mime_lang);
                }
                else
                {
                    stream << '(';
                    literal_out_str(stream, mime_lang);
                    stream << ' ';
                    send(stream, mime_lang_params);
                    stream << ')';
                }
            }
            stream << ' ';

            // location
            literal_out_str(stream, mime_location);
        } // extended
    }
};

using BodyStructurePtr = std::shared_ptr<BodyStructure>;

}
