#include "body_structure_loader.h"
#include "body_structure_actions.h"

#include <backend/envelope.h>
#include <backend/mbody/storage/message_storage.h>
#include <yplatform/find.h>

namespace yimap { namespace mbody {

BodystructureLoader::BodystructureLoader(
    const RawHeadersList& rawHeadersList,
    ImapContextPtr context,
    const std::string& stid,
    const macs::MimeParts& mimeParts)
    : mdsLoader(std::make_shared<BodystructureLoaderMDS>(rawHeadersList, context, stid, mimeParts))
    , context(context)
    , stid(stid)
{
}

FutureBodystructure BodystructureLoader::load()
{
    return mdsLoader->load();
}

BodystructureLoaderMDS::BodystructureLoaderMDS(
    const RawHeadersList& rawHeadersList,
    ImapContextPtr context,
    const string& stid,
    macs::MimeParts mimeParts)
    : stid(stid), rawHeadersList(rawHeadersList), context(context), mimeParts(std::move(mimeParts))
{
}

FutureBodystructure BodystructureLoaderMDS::load()
{
    Promise<BodyStructurePtr> promiseResult;

    auto self = shared_from_this();
    auto cb = [this, self, promiseResult](std::string err, std::string msg) mutable {
        if (!err.empty())
        {
            auto e = std::runtime_error(std::move(err));
            promiseResult.set_exception(std::make_exception_ptr(std::move(e)));
            return;
        }
        try
        {
            storage = std::make_shared<MessageStorageString>(
                std::make_shared<const std::string>(std::move(msg)), "", inlineLevel);
            auto result = std::make_shared<BodyStructure>();
            doLoad("1", *result);
            promiseResult.set(result);
        }
        catch (...)
        {
            promiseResult.set_exception(std::current_exception());
        }
    };

    auto messageStorage =
        yplatform::find<MessageStorageFactory>("message_storage")->create(context, stid, mimeParts);
    messageStorage->getWhole("1", std::move(cb));

    return promiseResult;
}

std::size_t BodystructureLoaderMDS::getBodyLines(const std::string& hid)
{
    try
    {
        auto bodyData = storage->getBody(hid);
        if (!bodyData)
        {
            throw std::runtime_error("no body");
        }
        auto lineCount = std::count(bodyData->begin(), bodyData->end(), '\n') + 1;
        if (!bodyData->empty() && bodyData->back() == '\n')
        {
            --lineCount;
        }
        return lineCount;
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("getBodyLines: ") + e.what(), hid);
    }
}

std::size_t BodystructureLoaderMDS::getBodySize(const std::string& hid, MetaAttributesPtr attrs)
{
    try
    {
        const auto lengthIter = attrs->find("length");
        if (lengthIter == attrs->end())
        {
            throw std::runtime_error("no length attr");
        }
        const auto length = std::stoull(lengthIter->second);
        return length;
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("getBodySize: ") + e.what(), hid);
    }
}

std::pair<std::string, std::string> BodystructureLoaderMDS::getContentType(
    const std::string& hid,
    MetaAttributesPtr attrs)
{
    try
    {
        auto ctypeIter = attrs->find("content_type.type");
        if (ctypeIter == attrs->end())
        {
            throw std::runtime_error("no content_type.type");
        }
        auto csubIter = attrs->find("content_type.subtype");
        if (csubIter == attrs->end())
        {
            throw std::runtime_error("no content_type.subtype");
        }
        return std::make_pair(ctypeIter->second, csubIter->second);
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("getContentType: ") + e.what(), hid);
    }
}

void BodystructureLoaderMDS::fillInlineMessageEnvelope(
    const std::string& hid,
    BodyStructure& bodyStructure)
{
    try
    {
        const auto metaLevel = storage->getBodyStruct(hid);
        // find first (and only) part of current node
        const auto part = metaLevel->begin();
        if (part != metaLevel->end())
        {
            EnvelopeData envelope;
            const auto part_headers = storage->getHeader(*part);
            if (!parse_envelope(*part_headers, envelope, rawHeadersList))
            {
                throw std::runtime_error("failed to parse envelope");
            }
            bodyStructure.envelope = std::move(envelope);
        }
        else
        {
            TASK_LOG(context, error) << "no part in metaLevel for stid " << stid;
            // throw std::runtime_error("no part in metaLevel"); // why commented?
        }
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("fillInlineMessageEnvelope: ") + e.what(), hid);
    }
}

void BodystructureLoaderMDS::fillHeaders(const std::string& hid, BodyStructure& bodyStructure)
{
    try
    {
        auto headers = storage->getHeader(hid);
        if (!headers)
        {
            throw std::runtime_error("no headers");
        }
        BodyStructureActions actions(context, bodyStructure);
        if (!rfc822::parseMessage(*headers, actions, rawHeadersList))
        {
            throw std::runtime_error("rfc822 parse error in" + *headers);
        }
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("fillHeaders: ") + e.what(), hid);
    }
}

void BodystructureLoaderMDS::doLoad(const string& hid, BodyStructure& bodyStructure)
{
    fillHeaders(hid, bodyStructure);

    MetaAttributesPtr attrs;
    try
    {
        attrs = storage->getHeaderStruct(hid);
        if (!attrs)
        {
            throw std::runtime_error("no headers struct");
        }
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("getHeaderStruct: ") + e.what(), hid);
    }

    bodyStructure.body_size = getBodySize(hid, attrs);

    std::string contentType, contentSubtype;
    std::tie(contentType, contentSubtype) = getContentType(hid, attrs);

    attrs.reset();

    if (contentType == "message" && contentSubtype == "rfc822")
    {
        fillInlineMessageEnvelope(hid, bodyStructure);
    }

    if (contentType == "text" || (contentType == "message" && contentSubtype == "rfc822"))
    {
        bodyStructure.body_lines = getBodyLines(hid);
    }

    bodyStructure.rfc_size = storage->getRfcSize();

    // loading body
    MetaLevelPtr metaLevel;
    try
    {
        metaLevel = storage->getBodyStruct(hid);
        if (!metaLevel)
        {
            throw std::runtime_error("no metalevel for bodystructure");
        }
    }
    catch (const std::exception& e)
    {
        throw BodystructureLoadingError(std::string("getBodyStruct: ") + e.what(), hid);
    }

    // recursively load nested parts
    for (auto& part : *metaLevel)
    {
        BodyStructure nestedPart;
        doLoad(part, nestedPart);
        bodyStructure.mime_parts.push_back(std::move(nestedPart));
    }
}

} // namespace mbody
} // namespace yimap
