#pragma once

#include "body_structure_data.h"
#include <macs/mime_part.h>
#include <common/imap_context.h>
#include <common/types.h>
#include <common/settings.h>
#include <common/log/logger.h>
#include <backend/backend_types.h>
#include <backend/mbody/storage/string_storage.h>

namespace yimap { namespace mbody {

class BodystructureLoaderMDS;

using FutureBodystructure = Future<BodyStructurePtr>;

class BodystructureLoader
{
public:
    using RawHeadersList = std::vector<std::string>;

    BodystructureLoader(
        const RawHeadersList& rawHeadersList,
        ImapContextPtr context,
        const std::string& stid,
        const macs::MimeParts& mimeParts);

    FutureBodystructure load();

private:
    std::shared_ptr<BodystructureLoaderMDS> mdsLoader;
    ImapContextPtr context;
    std::string stid;
};

class BodystructureLoaderMDS : public std::enable_shared_from_this<BodystructureLoaderMDS>
{
public:
    using RawHeadersList = BodystructureLoader::RawHeadersList;

    BodystructureLoaderMDS(
        const RawHeadersList& rawHeadersList,
        ImapContextPtr context,
        const string& srcStid,
        macs::MimeParts mimeParts);

    FutureBodystructure load();

protected:
    void doLoad(const string& hid, BodyStructure& bodyStructure);

    std::size_t getBodyLines(const std::string& hid);
    std::size_t getBodySize(const std::string& hid, MetaAttributesPtr attrs);
    std::pair<std::string, std::string> getContentType(
        const std::string& hid,
        MetaAttributesPtr attrs);
    void fillInlineMessageEnvelope(const std::string& hid, BodyStructure& bodyStructure);
    void fillHeaders(const std::string& hid, BodyStructure& bodyStructure);

    const std::string stid;
    const RawHeadersList& rawHeadersList;
    ImapContextPtr context;
    MessageStorageStringPtr storage;
    macs::MimeParts mimeParts;
    const int inlineLevel = 14; // max depth of inline messages. required by mimeparser
};

struct BodystructureLoadingError : public std::runtime_error
{
    explicit BodystructureLoadingError(const string& message, const string& hid = "")
        : runtime_error(string("BodystructureLoadingError: ") + message + " hid=" + hid)
    {
    }
};

} // namespace mbody
} // namespace yimap
