#include "message_loader.h"

#include <yplatform/find.h>
#include <yplatform/encoding/iterators/crlf.h>
#include <yplatform/log.h>
#include <yplatform/util/sstream.h>
#include <boost/lexical_cast.hpp>
#include <boost/range/iterator.hpp>

#include <vector>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

class GetHid : public std::enable_shared_from_this<GetHid>
{
public:
    using Handler = std::function<void(std::string err, std::string hid)>;
    using HeaderStruct = MessageStorage::HeaderStruct;
    GetHid(MessageStoragePtr messageStorage, std::string imapHid, Handler handler)
        : messageStorage(messageStorage)
        , imapHid(std::move(imapHid))
        , handler(std::move(handler))
        , hidStream(hid, this->imapHid.length() + 10)
    {
    }
    void operator()()
    {
        processNextPart();
    }

private:
    void processNextPart()
    {
        if (afterTheLastDot < imapHid.length())
        {
            auto nextDotPos = imapHid.find('.', afterTheLastDot);
            if (nextDotPos == std::string::npos)
            {
                nextDotPos = imapHid.length();
            }
            hidStream << "." << imapHid.substr(afterTheLastDot, nextDotPos - afterTheLastDot);
            auto cb = std::bind(&GetHid::headerStructCallback, shared_from_this(), ph::_1, ph::_2);
            afterTheLastDot = nextDotPos + 1;
            messageStorage->getHeaderStruct(hid, std::move(cb));
        }
        else
        {
            handler("", std::move(hid));
        }
    }
    void headerStructCallback(std::string err, HeaderStruct headerStruct)
    {
        if (!err.empty())
        {
            handler(std::move(err), "");
            return;
        }
        if (headerStruct.contentType == "message" && headerStruct.contentSubtype == "rfc822")
        {
            hidStream << ".1";
        }
        processNextPart();
    }
    MessageStoragePtr messageStorage;
    std::string imapHid;
    Handler handler;
    std::string hid = "1";
    std::size_t afterTheLastDot = 0;
    yplatform::sstream hidStream; // stream to this->hid
};

MessageLoader::MessageLoader(
    ImapContextPtr context,
    const std::string& stid,
    const macs::MimeParts& mimeParts)
    : stid(stid), context(context)
{
    auto factory = yplatform::find<MessageStorageFactory>("message_storage");
    messageStorage = factory->create(context, stid, mimeParts);
}

void MessageLoader::load(
    const std::string& part,
    bool withHeader,
    bool withBody,
    const handler_type& handler)
{
    Future<Result> future = loadFromMDS(part, withHeader, withBody);
    auto self = shared_from_this();
    future.add_callback([this, self, future, handler]() mutable {
        Result res;
        try
        {
            res = future.get();
        }
        catch (const std::exception& e)
        {
            res.err = std::string("MessageLoader::load error in handler: ") + e.what();
        }
        if (!res.err.empty())
        {
            handler(res.err, {});
            return;
        }
        handler({}, std::make_shared<std::string>(std::move(res.msg)));
    });
}

Future<MessageLoader::Result> MessageLoader::loadFromMDS(
    const std::string& part,
    bool withHeader,
    bool withBody)
{
    Promise<Result> promise;
    auto cb = std::bind(
        &MessageLoader::doLoadFromMDS,
        shared_from_this(),
        promise,
        part,
        withHeader,
        withBody,
        ph::_1,
        ph::_2);
    auto getHid = std::make_shared<GetHid>(messageStorage, part, std::move(cb));
    (*getHid)();
    return promise;
}

void MessageLoader::doLoadFromMDS(
    Promise<Result> promise,
    const std::string& part,
    bool withHeader,
    bool withBody,
    std::string err,
    std::string hid)
{
    // Some client just can't get that there are some emails without 1.1 mime part
    if (!err.empty() && part == "1")
    {
        doLoadFromMDS(promise, "", withHeader, withBody, "", "1");
        return;
    }
    if (!err.empty())
    {
        Result result = { "", 0, std::move(err) };
        promise.set(std::move(result));
        return;
    }
    auto self = shared_from_this();
    auto cb = [this, self, promise](std::string err, std::string msg) mutable {
        if (!err.empty())
        {
            Result result = { "", 0, std::move(err) };
            promise.set(std::move(result));
            return;
        }
        auto getSizeHandler =
            [this, self, promise, msg = std::move(msg)](std::string err, std::size_t size) mutable {
                Result result = { std::move(msg), 0, "" };
                if (err.empty())
                {
                    result.rfcSize = size;
                }
                promise.set(std::move(result));
            };
        messageStorage->getMessageSize(std::move(getSizeHandler));
    };
    if (withHeader && withBody)
    {
        messageStorage->getWhole(std::move(hid), std::move(cb));
    }
    else if (withHeader)
    {
        messageStorage->getHeaders(std::move(hid), std::move(cb));
    }
    else if (withBody)
    {
        messageStorage->getBody(std::move(hid), std::move(cb));
    }
}

string MessageLoader::prepareHid(const std::string& imapHid)
{
    // hids in mail metabase are prefixed by "1."
    return imapHid.size() ? "1." + imapHid : string("1");
}

} // namespace mbody
} // namespace yimap
