#pragma once

#include <backend/mbody/types.h>
#include <backend/mbody/storage/string_storage.h>
#include <butil/butil.h>
#include <yplatform/util/sstream.h>
#include <boost/algorithm/string.hpp>

namespace yimap { namespace mbody {

class InlineMessageStorage : public std::enable_shared_from_this<InlineMessageStorage>
{
public:
    template <typename... Result>
    using Handler = std::function<void(std::string err, Result...)>;
    using BodyGetter = std::function<void(std::string hid, Handler<std::string>)>;

    InlineMessageStorage(ConstMetaPartsPtr meta, BodyGetter bodyGetter)
        : meta(meta), bodyGetter(std::move(bodyGetter))
    {
        inlm = buildInlineMessages();
    }

    void getHeaderStruct(const std::string& hid, Handler<MetaAttributesPtr> handler)
    {
        auto self = shared_from_this();
        auto cb = [this, self, hid, handler = std::move(handler)](
                      std::string err, MessageStorageStringPtr storage, std::size_t sawoff) {
            if (!err.empty())
            {
                handler(std::move(err), nullptr);
                return;
            }
            try
            {
                auto headerStruct = storage->getHeaderStruct(hid.substr(sawoff));
                if (!headerStruct)
                {
                    throw std::runtime_error(
                        "can't get HeaderStruct for inline message hid=" + hid);
                }
                handler("", headerStruct);
            }
            catch (const std::exception& e)
            {
                handler(e.what(), nullptr);
            }
        };
        findInlineMessage(hid, std::move(cb));
    }

    void getPart(
        const std::string& hid,
        bool withHeaders,
        bool withBody,
        Handler<std::string> handler)
    {
        auto self = shared_from_this();
        auto cb = [this, self, hid, withHeaders, withBody, handler = std::move(handler)](
                      std::string err, MessageStorageStringPtr storage, std::size_t sawoff) {
            if (!err.empty())
            {
                handler(std::move(err), "");
                return;
            }
            try
            {
                ConstStringPtr header, body;
                if (withHeaders)
                {
                    header = storage->getHeader(hid.substr(sawoff));
                    if (!header)
                    {
                        throw std::runtime_error("can't get header from MessageStringStorage");
                    }
                }
                if (withBody)
                {
                    body = storage->getBody(hid.substr(sawoff));
                    if (!body)
                    {
                        throw std::runtime_error("can't get body from MessageStringStorage");
                    }
                }
                std::string res;
                yplatform::sstream str(
                    res, (withHeaders ? header->length() : 0) + (withBody ? body->length() : 0));
                if (withHeaders)
                {
                    str << *header;
                }
                if (withBody)
                {
                    str << *body;
                }
                handler("", std::move(res));
            }
            catch (const std::exception& e)
            {
                handler(e.what(), "");
            }
        };
        findInlineMessage(hid, std::move(cb));
    }

private:
    InlineMessages buildInlineMessages()
    {
        InlineMessages res;
        for (const auto part : *meta)
        {
            auto contentType = part.second.find("content_type.type");
            if (contentType == part.second.end())
            {
                continue;
            }
            auto contentSubtype = part.second.find("content_type.subtype");
            if (contentSubtype == part.second.end())
            {
                continue;
            }
            if (boost::iequals(contentType->second, "message") &&
                boost::iequals(contentSubtype->second, "rfc822"))
            {
                res[part.first] = nullptr;
            }
        }
        return res;
    }

    void getInlineMessage(const std::string& hid, Handler<std::string> handler)
    {
        auto cb = std::bind(
            &InlineMessageStorage::bodyGetterHandler,
            shared_from_this(),
            hid,
            std::move(handler),
            std::placeholders::_1,
            std::placeholders::_2);
        bodyGetter(hid, std::move(cb));
    }

    void bodyGetterHandler(
        const std::string& hid,
        const Handler<std::string>& handler,
        std::string err,
        std::string body)
    {
        if (!err.empty())
        {
            handler("can't get inline message: " + std::move(err), "");
            return;
        }
        MetaParts::const_iterator it = meta->find(hid);
        if (it != meta->end())
        {
            const MetaAttributes& attributes = it->second;
            MetaAttributes::const_iterator attrIt = attributes.find("content_transfer_encoding");
            if (attrIt != attributes.end())
            {
                const std::string& transferEncoding = attrIt->second;
                if (boost::algorithm::iequals(transferEncoding, "base64"))
                {
                    handler("", decode_base64(body));
                    return;
                }
                else if (boost::algorithm::iequals(transferEncoding, "quoted-printable"))
                {
                    handler("", decode_qp(body));
                    return;
                }
            }
        }
        handler("", std::move(body));
    };

    void findInlineMessage(
        const std::string& hid,
        Handler<MessageStorageStringPtr, std::size_t> handler)
    {
        for (auto inlineMessage = inlm.begin(); inlineMessage != inlm.end(); inlineMessage++)
        {
            if (inlineMessage->first.size() < hid.size() &&
                !strncmp(inlineMessage->first.c_str(), hid.c_str(), inlineMessage->first.size()) &&
                '.' == hid[inlineMessage->first.size()])
            {
                const auto sawoff = inlineMessage->first.size() + 1;
                auto self = shared_from_this();
                auto cb = [this,
                           self,
                           handler = std::move(handler),
                           sawoff,
                           prefix = inlineMessage->first + "."](std::string err, std::string msg) {
                    if (!err.empty())
                    {
                        handler(std::move(err), nullptr, 0);
                        return;
                    }
                    auto content = std::make_shared<const std::string>(std::move(msg));
                    auto storage =
                        std::make_shared<MessageStorageString>(content, prefix, inlineLevel);
                    handler("", storage, sawoff);
                };
                getInlineMessage(inlineMessage->first, std::move(cb));
                return;
            }
        }
        handler("can't find inline msg with hid " + hid, nullptr, 0);
    }

    ConstMetaPartsPtr meta;
    InlineMessages inlm;
    BodyGetter bodyGetter;
    const int inlineLevel = 14; // max depth of inline messages. required by mimeparser
};

} // namespace mbody
} // namespace yimap
