#include "message_storage.h"

#include <yplatform/module.h>
#include <yplatform/util/sstream.h>
#include <macs/types.h>
#include <backend/mbody/storage/raw_storage.h>
#include <backend/mbody/storage/inline_message_storage.h>
#include <backend/mbody/storage/mime_storage.h>
#include <backend/mbody/storage/messages_cache.h>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

template <typename RawStorage>
class MessageStorageImpl
    : public MessageStorage
    , public std::enable_shared_from_this<MessageStorageImpl<RawStorage>>
{
public:
    MessageStorageImpl(RawStorage rawStorage, const MimeStoragePtr& mimeStorage)
        : storage(rawStorage), mimeStorage(mimeStorage)
    {
    }

    void getWhole(std::string hid, Handler<std::string> handler) const override
    {
        auto cb = std::bind(
            &MessageStorageImpl::getHeadersCallback,
            yplatform::shared_from(this),
            std::move(hid),
            true,
            std::move(handler),
            ph::_1,
            ph::_2);
        mimeStorage->get(std::move(cb));
    }

    void getHeaders(std::string hid, Handler<std::string> handler) const override
    {
        auto cb = std::bind(
            &MessageStorageImpl::getHeadersCallback,
            yplatform::shared_from(this),
            std::move(hid),
            false,
            std::move(handler),
            ph::_1,
            ph::_2);
        mimeStorage->get(std::move(cb));
    }

    void getBody(std::string hid, Handler<std::string> handler) const override
    {
        auto cb = std::bind(
            &MessageStorageImpl::getBodyCallback,
            yplatform::shared_from(this),
            std::move(hid),
            std::move(handler),
            ph::_1,
            ph::_2);
        mimeStorage->get(std::move(cb));
    }

    void getHeaderStruct(std::string hid, Handler<HeaderStruct> handler) const override
    {
        auto cb = std::bind(
            &MessageStorageImpl::getHeaderStructCallback,
            yplatform::shared_from(this),
            std::move(hid),
            std::move(handler),
            ph::_1,
            ph::_2);
        mimeStorage->get(std::move(cb));
    }

    void getMessageSize(Handler<std::size_t> handler) const override
    {
        auto cb = [this, self = yplatform::shared_from(this), handler = std::move(handler)](
                      std::string err, MimeStorage::MetaData data) {
            if (!err.empty())
            {
                handler(std::move(err), 0);
            }
            else
            {
                handler("", data.messageLength);
            }
        };
        mimeStorage->get(std::move(cb));
    }

private:
    void getHeadersCallback(
        std::string hid,
        bool withBody,
        Handler<std::string> handler,
        std::string err,
        MimeStorage::MetaData data) const
    {
        if (!err.empty())
        {
            handler("MessageStorageImpl can't get headers: " + err, "");
            return;
        }
        try
        {
            auto mpi = data.metaParts->find(hid);
            if (mpi != data.metaParts->end())
            {
                auto bodyOffset = std::stoull(mpi->second.at("offset"));
                auto bodyLength = std::stoull(mpi->second.at("length"));
                if (mpi != data.metaParts->begin())
                {
                    --mpi;
                    auto prevBodyOffset = std::stoull(mpi->second.at("offset"));
                    auto prevBodyLength = std::stoull(mpi->second.at("length"));
                    auto self = yplatform::shared_from(this);
                    auto cb = std::bind(
                        &MessageStorageImpl::truncateHeaders,
                        self,
                        std::move(handler),
                        bodyOffset,
                        ph::_1,
                        ph::_2);
                    std::size_t offsetBegin;
                    if (std::equal(
                            hid.begin(),
                            hid.begin() + std::min(hid.length(), mpi->first.length()),
                            mpi->first.begin()))
                    {
                        // if prev part contains required part (e.g. hid = 1.2.1, prev part hid
                        // = 1.2)
                        if (prevBodyOffset < 2)
                        {
                            handler(
                                std::string("body offset should be at least 2 hid=") + mpi->first,
                                "");
                            return;
                        }
                        offsetBegin = data.xmlLength + prevBodyOffset - 1;
                    }
                    else
                    {
                        // if prev part precede required part (e.g. hid = 1.2.2, prev part hid
                        // = 1.2.1)
                        offsetBegin = data.xmlLength + prevBodyOffset + prevBodyLength;
                    }
                    auto offsetEnd = data.xmlLength + bodyOffset + (withBody ? bodyLength : 0);
                    storage->get(offsetBegin, offsetEnd - offsetBegin, std::move(cb));
                }
                else
                {
                    auto cb = std::bind(
                        &MessageStorageImpl::truncateXml,
                        yplatform::shared_from(this),
                        std::move(handler),
                        ph::_1,
                        ph::_2);
                    storage->get(
                        data.xmlLength, bodyOffset + (withBody ? bodyLength : 0), std::move(cb));
                }
            }
            else
            {
                auto bodyGetter = std::bind(
                    &MessageStorage::getBody, yplatform::shared_from(this), ph::_1, ph::_2);
                auto inlineRepo =
                    std::make_shared<InlineMessageStorage>(data.metaParts, std::move(bodyGetter));
                inlineRepo->getPart(hid, true, withBody, std::move(handler));
            }
        }
        catch (const std::exception& e)
        {
            handler(std::string("getHeadersCallback: ") + e.what(), "");
        }
    }

    void getBodyCallback(
        std::string hid,
        Handler<std::string> handler,
        std::string err,
        MimeStorage::MetaData data) const
    {
        if (!err.empty())
        {
            handler("MessageStorageImpl can't get body: " + err, "");
            return;
        }
        try
        {
            auto mpi = data.metaParts->find(hid);
            if (mpi != data.metaParts->end())
            {
                auto offset = std::stoull(mpi->second.at("offset")) + data.xmlLength;
                auto length = std::stoull(mpi->second.at("length"));
                storage->get(offset, length, std::move(handler));
            }
            else
            {
                auto bodyGetter = std::bind(
                    &MessageStorage::getBody, yplatform::shared_from(this), ph::_1, ph::_2);
                auto inlineRepo =
                    std::make_shared<InlineMessageStorage>(data.metaParts, std::move(bodyGetter));
                inlineRepo->getPart(hid, false, true, std::move(handler));
            }
        }
        catch (const std::exception& e)
        {
            handler(std::string("getBodyCallback: ") + e.what(), "");
        }
    }

    void getHeaderStructCallback(
        std::string hid,
        Handler<HeaderStruct> handler,
        std::string err,
        MimeStorage::MetaData data) const
    {
        if (!err.empty())
        {
            handler(std::move(err), HeaderStruct());
            return;
        }
        try
        {
            auto it = data.metaParts->find(hid);
            if (it != data.metaParts->end())
            {
                HeaderStruct result;
                result.contentType = it->second.at("content_type.type");
                result.contentSubtype = it->second.at("content_type.subtype");
                handler("", std::move(result));
            }
            else
            {
                auto bodyGetter = std::bind(
                    &MessageStorage::getBody, yplatform::shared_from(this), ph::_1, ph::_2);
                auto cb = [this, self = yplatform::shared_from(this), handler = std::move(handler)](
                              std::string err, MetaAttributesPtr meta) {
                    if (!err.empty())
                    {
                        handler(err, HeaderStruct());
                        return;
                    }
                    HeaderStruct result;
                    result.contentType = (*meta)["content_type.type"];
                    result.contentSubtype = (*meta)["content_type.subtype"];
                    handler("", std::move(result));
                };
                auto inlineRepo =
                    std::make_shared<InlineMessageStorage>(data.metaParts, std::move(bodyGetter));
                inlineRepo->getHeaderStruct(hid, std::move(cb));
            }
        }
        catch (const std::exception& e)
        {
            handler(std::string("getHeaderStruct: ") + e.what(), HeaderStruct());
        }
    }

    void truncateXml(Handler<std::string> handler, std::string err, std::string msg) const
    {
        if (!err.empty())
        {
            handler(std::move(err), "");
            return;
        }
        if (std::equal(
                msg.begin(),
                msg.begin() + std::min(msg.length(), XML_PREFIX.length()),
                XML_PREFIX.begin()))
        {
            // if msg begins with XML_PREFIX
            auto xmlEndPos = msg.find(XML_SUFFIX);
            if (xmlEndPos == std::string::npos)
            {
                std::string err_text;
                const auto hint = sizeof("message begins with \"\" but dont contains \"\": "
                                         "xxxxxxxxxxxxxxxxxxxx...") +
                    XML_PREFIX.length() + XML_SUFFIX.length();
                yplatform::sstream str(err_text, hint);
                str << "message begins with \"" << XML_PREFIX << "\" but dont contains \""
                    << XML_SUFFIX << "\": " << msg.substr(20) << "...";
                handler(std::move(err_text), "");
                return;
            }
            xmlEndPos = msg.find('\n', xmlEndPos);
            if (xmlEndPos == std::string::npos)
            {
                handler("MessageStorageImpl::truncateXml: Can't find line ending after xml", "");
                return;
            }
            ++xmlEndPos;
            handler("", msg.substr(xmlEndPos)); // return message without xml
        }
        else
        {
            handler("", std::move(msg)); // no xml in the begining -> return as is
        }
    }

    void truncateHeaders(
        Handler<std::string> handler,
        std::size_t bodyOffset,
        std::string err,
        std::string body) const
    {
        if (!err.empty())
        {
            handler("MessageStorageImpl can't get headers" + err, "");
            return;
        }
        auto beg = body.rfind("\n--", bodyOffset);
        if (beg == std::string::npos)
        {
            handler("MessageStorageImpl can't find boundary", "");
            return;
        }
        beg = body.find('\n', ++beg);
        if (beg == std::string::npos)
        {
            handler("MessageStorageImpl can't find boundary", "");
            return;
        }
        ++beg;
        handler("", body.substr(beg));
    }

    RawStorage storage;
    MimeStoragePtr mimeStorage;
};

class MessageStorageFactoryImpl
    : public MessageStorageFactory
    , public yplatform::module
{
public:
    MessageStorageFactoryImpl(boost::asio::io_service& io, const Ptree& config) : io(io)
    {
        StorageSettings opt;
        opt.extraAttrs = config.get<std::string>("attrs", "");
        opt.stidTruncatePrefix = config.get<std::string>("stidTruncatePrefix", "");
        opt.cacheSize = config.get<size_t>("cacheSize");
        settings = std::make_shared<const StorageSettings>(std::move(opt));
        cache = std::make_shared<MessagesCache>(opt.cacheSize);
    }

    MessageStoragePtr create(
        ImapContextPtr context,
        const std::string& stid,
        const macs::MimeParts& mimeParts) const override
    {
        if (!settings)
        {
            throw std::runtime_error("settings not inited");
        }
        auto rawStorage = createRawStorage(io, settings, context, stid, cache);
        auto mimeStorage = std::make_shared<MimeStorage>(mimeParts);
        auto messageStorage =
            std::make_shared<MessageStorageImpl<decltype(rawStorage)>>(rawStorage, mimeStorage);
        return messageStorage;
    }

private:
    StorageSettingsConstPtr settings;
    boost::asio::io_service& io;
    std::shared_ptr<MessagesCache> cache;
};

} // namespace mbody
} // namespace yimap

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yimap::mbody::MessageStorageFactoryImpl)
