#pragma once

#include <common/types.h>
#include <yplatform/algorithm/lru_cache.h>

namespace yimap { namespace mbody {

class MessagesCache
{
public:
    MessagesCache(size_t capacity)
        : cache(capacity, [](auto /*key*/, auto val) { return val->length(); }), capacity(capacity)
    {
    }

    StringPtr get(const string& stid, size_t offset, size_t length)
    {
        Key key{ stid, offset, length };
        auto res = cache.get(key);
        if (!res) return {};
        return *res;
    }

    void put(const string& stid, size_t offset, size_t length, const string& message)
    {
        // optimization to avoid unnecessary message copy
        if (message.length() > capacity) return;
        Key key{ stid, offset, length };
        cache.put(key, std::make_shared<string>(message));
    }

private:
    struct Key
    {
        string stid;
        size_t offset = 0;
        size_t length = 0;

        bool operator==(const Key& other) const
        {
            return stid == other.stid && offset == other.offset && length == other.length;
        }

        friend size_t hash_value(const MessagesCache::Key& key)
        {
            size_t seed = 0;
            boost::hash<string> stringHasher;
            boost::hash_combine(seed, stringHasher(key.stid));
            boost::hash<size_t> numHasher;
            boost::hash_combine(seed, numHasher(key.offset));
            boost::hash_combine(seed, numHasher(key.length));
            return seed;
        }
    };

    using CacheEntrySizeFunc = std::function<size_t(Key, StringPtr)>;
    using LruCache = yplatform::lru_cache<Key, StringPtr, CacheEntrySizeFunc>;
    LruCache cache;
    size_t capacity;
};

}}
