#pragma once

#include <boost/optional.hpp>
#include <yplatform/util/sstream.h>
#include <macs/types.h>

namespace yimap { namespace mbody {

namespace ph = std::placeholders;

static const std::string XML_PREFIX = "<?xml";
static const std::string XML_SUFFIX = "</message>";

class MimeStorage : public std::enable_shared_from_this<MimeStorage>
{
public:
    using HeaderStruct = MessageStorage::HeaderStruct;

    struct MetaData
    {
        std::size_t xmlLength;
        std::size_t messageLength;
        ConstMetaPartsPtr metaParts;
    };

    using Handler = std::function<void(std::string err, MetaData)>;

    MimeStorage(const macs::MimeParts& mimeParts)
    {
        if (!mimeParts.empty())
        {
            metaData = convert(mimeParts);
        }
    }

    void get(Handler handler)
    {
        auto ret = meta();
        if (ret)
        {
            handler("", std::move(*ret));
        }
        else
        {
            handler("no mime structure for message", MetaData());
        }
    }

private:
    boost::optional<MetaData> meta()
    {
        std::lock_guard<std::mutex> lock(mutex);
        return metaData;
    }

    void setMeta(MetaData data)
    {
        std::lock_guard<std::mutex> lock(mutex);
        metaData = data;
    }

    MetaData convert(const macs::MimeParts& mimeParts)
    {
        MetaParts parts;
        for (const auto& part : mimeParts)
        {
            MetaAttributes attributes;
            attributes["offset"] = std::to_string(part.offsetBegin());
            attributes["length"] = std::to_string(part.length());
            attributes["content_type.type"] = part.contentType();
            attributes["content_type.subtype"] = part.contentSubtype();
            attributes["content_type.name"] = part.name();
            attributes["content_type.charset"] = part.charset();
            attributes["content_transfer_encoding"] = part.encoding();
            attributes["content_disposition.value"] = part.contentDisposition();
            attributes["content_disposition.filename"] = part.fileName();
            attributes["content_id"] = part.cid();
            parts.emplace(part.hid(), std::move(attributes));
        }
        MetaData result;
        // when MetaData taken from macs::MimeParts all offsets are incremented on xmlLength
        // so in this case we assume that xmlLength and messageLength equals 0
        result.xmlLength = 0;
        result.messageLength = 0;
        result.metaParts = std::make_shared<const MetaParts>(std::move(parts));
        return result;
    }

    std::mutex mutex;
    boost::optional<MetaData> metaData;
};

using MimeStoragePtr = std::shared_ptr<MimeStorage>;

} // namespace mbody
} // namespace yimap
