#pragma once

#include <backend/mbody/types.h>
#include <backend/mbody/storage/messages_cache.h>
#include <common/imap_context.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/find.h>
#include <yplatform/util/sstream.h>
#include <boost/algorithm/string.hpp>
#include <boost/optional.hpp>

namespace yimap { namespace mbody {

using LengthOpt = boost::optional<std::size_t>;

namespace ph = std::placeholders;

template <typename HTTPClient, typename RateController, typename Cache>
class RawStorage
    : public std::enable_shared_from_this<RawStorage<HTTPClient, RateController, Cache>>
{
    using Handler = std::function<void(std::string err, std::string body)>;

public:
    RawStorage(
        boost::asio::io_service& io,
        const StorageSettingsConstPtr& settings,
        ImapContextPtr context,
        const std::string& stid,
        HTTPClient httpClient,
        RateController rateController,
        Cache cache)
        : io(io)
        , settings(settings)
        , warningLog(YGLOBAL_LOG_SERVICE, "warning_log")
        , context(context)
        , stid(truncateStid(stid, settings->stidTruncatePrefix))
        , httpClient(httpClient)
        , rateController(rateController)
        , cache(cache)
    {
    }

    void get(Handler handler)
    {
        get(0, LengthOpt(), std::move(handler));
    }

    void get(const std::size_t offset, LengthOpt length, Handler handler)
    {
        if (length && *length == 0)
        {
            return handler(std::string(), std::string());
        }

        RequestData data;
        data.offset = offset;
        data.length = length;
        data.handler = std::move(handler);
        io.post([this, self = yplatform::shared_from(this), data] {
            if (auto message = getFromCache(data))
            {
                updateStatsOnCacheHit();
                data.handler("", *message);
            }
            else
            {
                updateStatsOnCacheMiss();
                doRequest(std::move(data));
            }
        });
    }

private:
    struct RequestData
    {
        size_t offset;
        LengthOpt length;
        Handler handler;
    };

    StringPtr getFromCache(const RequestData& data)
    {
        return cache->get(stid, data.offset, data.length ? *data.length : 0);
    }

    void doRequest(RequestData data)
    {
        auto self = yplatform::shared_from(this);
        rateController->post([data = std::move(data), self, this](auto&& ec, auto&& rcHandler) {
            if (ec)
            {
                YLOG(warningLog, warning) << "Raw storage rc error: " << ec.message();
                rcHandler();
                data.handler(ec.message(), "");
                return;
            }

            auto request = yhttp::request::GET(makeUrl(), makeHeaders(data.offset, data.length));
            httpClient->async_run(
                context,
                std::move(request),
                io.wrap([this, self, data, rcHandler](
                            ErrorCode ec, yhttp::response response) mutable {
                    rcHandler();
                    if (ec)
                    {
                        data.handler(ec.message(), "");
                    }
                    else if (response.status / 100 != 2)
                    {
                        std::string msg;
                        yplatform::sstream str(
                            msg, response.reason.length() + sizeof("bad response code: xxx "));
                        str << "bad response code: " << response.status << " " << response.reason;
                        data.handler(std::move(msg), "");
                    }
                    else
                    {
                        putToCache(data, response.body);
                        data.handler("", std::move(response.body));
                    }
                }));
        });
    }

    void updateStatsOnCacheHit()
    {
        context->messagesStats.mbodyCacheHitCount++;
    }

    void updateStatsOnCacheMiss()
    {
        context->messagesStats.storageRequestsCount++;
    }

    std::string makeUrl()
    {
        std::vector<std::string> attrs;
        if (!settings->extraAttrs.empty())
        {
            attrs.push_back(settings->extraAttrs);
        }

        std::string url;
        auto hint = stid.length() + settings->extraAttrs.length() + sizeof("/gate/get/?&&&&");
        yplatform::sstream str(url, hint);
        str << "/gate/get/" << stid;
        if (!attrs.empty())
        {
            str << "?" << boost::join(attrs, "&");
        }
        return url;
    }

    std::string makeHeaders(const std::size_t offset, LengthOpt length)
    {
        std::string res;
        yplatform::sstream str(res, sizeof("Range: bytes=1000000500-2000000500\r\n"));

        str << "Range: bytes=" << offset << "-";
        if (length)
        {
            str << offset + *length - 1;
        }
        str << "\r\n";

        return res;
    }

    std::string truncateStid(const std::string& stid, const std::string& prefix)
    {
        if (!prefix.empty() && stid.length() >= prefix.length() &&
            std::equal(prefix.begin(), prefix.end(), stid.begin()))
        {
            // erase prefix
            return stid.substr(prefix.length());
        }
        else
        {
            return stid;
        }
    }

    void putToCache(const RequestData& data, const string& message)
    {
        cache->put(stid, data.offset, data.length ? *data.length : 0, message);
    }

    boost::asio::io_service& io;
    StorageSettingsConstPtr settings;
    mutable yplatform::log::source warningLog;
    ImapContextPtr context;
    const std::string stid;
    HTTPClient httpClient;
    RateController rateController;
    Cache cache;
};

inline auto createRawStorage(
    boost::asio::io_service& io,
    const StorageSettingsConstPtr& settings,
    ImapContextPtr context,
    const std::string& stid,
    std::shared_ptr<MessagesCache> cache)
{
    auto httpClient = yplatform::find<yhttp::cluster_client, std::shared_ptr>("storage_client");
    auto rcModule = yplatform::find<ymod_ratecontroller::rate_controller_module>("rate_controller");
    auto controller = rcModule->get_controller("mulcagate");
    return std::make_shared<RawStorage<
        std::shared_ptr<yhttp::cluster_client>,
        ymod_ratecontroller::rate_controller_ptr,
        std::shared_ptr<MessagesCache>>>(
        io, settings, context, stid, httpClient, controller, cache);
}

} // namespace mbody
} // namespace yimap
