#include "string_storage.h"

#include <butil/StrUtils/ToLower.h>
#include <boost/lexical_cast.hpp>

namespace yimap { namespace mbody {

MessageStorageString::MessageStorageString(
    ConstStringPtr _content,
    const std::string& _prefix,
    int _inlineLevel)
    : part(*_content, 0)
    , content(_content)
    , prefix(_prefix)
    , inlineLevel(_inlineLevel)
    , parsed(false)
{
}

MessageStorageString::~MessageStorageString()
{
}

int MessageStorageString::doParse()
{
    if (!parsed)
    {
        if (content)
        {
            mulca_mime::part::parse(inlineLevel, 0, content->size());
            splitToParts(*this, "1");
        }
        else
        {
            std::ostringstream s;
            s << __PRETTY_FUNCTION__ << ": "
              << "got NULL instead of std::string pointer";
        }
        parsed = true;
    }
    return !content;
}

void MessageStorageString::splitToParts(const part& p, const std::string& id)
{
    MetaAttributes ma;
    {
        std::ostringstream wrk;
        wrk << p.offset();
        ma["offset"] = wrk.str().c_str();
    }
    {
        std::ostringstream wrk;
        wrk << p.length();
        ma["length"] = wrk.str().c_str();
    }
    {
        std::ostringstream wrk;
        wrk << p.header.offset();
        ma["hdr_offset"] = wrk.str().c_str();
    }
    {
        std::ostringstream wrk;
        wrk << p.header.length();
        ma["hdr_length"] = wrk.str().c_str();
    }
    if (!strcasecmp(p.header.content_type.type.c_str(), "message") &&
        !strcasecmp(p.header.content_type.subtype.c_str(), "rfc822"))
    {
        if (p.inlineLevel)
        {
            // inlm[id]=NULL;
        }
        else
        {
            ma["link_only"] = "yes";
        }
    }
#define ADDFIELD(f)                                                                                \
    if (!p.header.f.empty())                                                                       \
    {                                                                                              \
        ma[#f] = p.header.f;                                                                       \
    }
    ADDFIELD(content_type.type);
    ADDFIELD(content_type.subtype);
    ADDFIELD(content_type.charset);
    ADDFIELD(content_type.name);
    ADDFIELD(content_transfer_encoding);
    ADDFIELD(content_disposition.value);
    ADDFIELD(content_disposition.filename);
    ADDFIELD(content_id);
    MetaAttributes::iterator ai;
    for (ai = ma.begin(); ai != ma.end(); ai++)
    {
        size_t epos = 0;
        while ((epos = ai->second.find('\r', epos)) != std::string::size_type(-1))
            ai->second.erase(epos, 1);
        epos = 0;
        while ((epos = ai->second.find('\n', epos)) != std::string::size_type(-1))
            ai->second.erase(epos, 1);
    }
    ToLower lowercase;
    if ((ai = ma.find("content_type.type")) != ma.end())
        std::transform(ai->second.begin(), ai->second.end(), ai->second.begin(), lowercase);
    if ((ai = ma.find("content_type.subtype")) != ma.end())
        std::transform(ai->second.begin(), ai->second.end(), ai->second.begin(), lowercase);
    meta[id] = ma;
    int idAddon = 1;
    for (mulca_mime::parts_t::const_iterator i = p.parts.begin(); i != p.parts.end();
         i++, idAddon++)
    {
        std::ostringstream wrk;
        wrk << id << "." << idAddon;
        splitToParts(*i, wrk.str().c_str());
    }
}

MetaAttributesPtr MessageStorageString::getHeaderStruct(const std::string& hid)
{
    if (doParse())
    {
        std::ostringstream s;
        s << "Can't parse the message. hid=" << prefix << hid;
        return MetaAttributesPtr();
    }

    MetaParts::iterator mpi;

    if ((mpi = meta.find(hid)) != meta.end())
    {
        MetaAttributesPtr res(new MetaAttributes(mpi->second));
        (*res)["hid"] = prefix + hid;
        return res;
    }

    std::ostringstream s;
    s << "Can't find the header. hid=" << hid;
    return MetaAttributesPtr();
}

MetaLevelPtr MessageStorageString::getBodyStruct(const std::string& hid)
{
    if (doParse()) return MetaLevelPtr();

    MetaLevelPtr res(new MetaLevel);
    int i;
    for (i = 1;; i++)
    {
        std::ostringstream wrk;
        wrk << hid << "." << i;
        std::string subhid(wrk.str());
        if (meta.find(subhid) != meta.end()) res->insert(res->end(), prefix + subhid);
        else
            break;
    }

    return res;
}

StringPtr MessageStorageString::getHeader(const std::string& hid)
{
    if (doParse()) return StringPtr();

    static const std::string OFFSET("hdr_offset");
    static const std::string LENGTH("hdr_length");

    StringPtr result;

    auto mpi = meta.find(hid);
    if (mpi != meta.end())
    {
        auto offset = boost::lexical_cast<size_t>(mpi->second[OFFSET]);
        auto length = boost::lexical_cast<size_t>(mpi->second[LENGTH]);
        result.reset(new std::string(content->substr(offset, length)));
    }

    return result;
}

StringPtr MessageStorageString::getBody(const std::string& hid)
{
    if (doParse()) return StringPtr();

    MetaParts::iterator mpi;
    const std::string OFFSET("offset"), LENGTH("length");
    mpi = meta.find(hid);
    if (mpi != meta.end())
        return StringPtr(new string(content->substr(
            boost::lexical_cast<int>(mpi->second[OFFSET].c_str()),
            boost::lexical_cast<int>(mpi->second[LENGTH].c_str()))));
    else
        return StringPtr();
    // NOT REACHED
    return StringPtr();
}

StringPtr MessageStorageString::getWhole()
{
    if (!content) return StringPtr();

    return StringPtr(new string(*content));
}

size_t MessageStorageString::getRfcSize() const
{
    if (!content)
    {
        return 0;
    }
    return content->length();
}

//-----------------------------------------------------------------------------

bool asNumCmp::operator()(const std::string& s1, const std::string& s2) const
{
    if (s1 == s2) return false;
    std::string::size_type b1 = 0, b2 = 0;
    while (true)
    {
        if (b1 >= s1.size() || b2 >= s2.size()) return (b1 >= s1.size()) && (b2 < s2.size());
        std::string::size_type e1, e2;
        e1 = s1.find('.', b1);
        e2 = s2.find('.', b2);
        std::string p1 = s1.substr(b1, e1 - b1);
        std::string p2 = s2.substr(b2, e2 - b2);
        if (p1 != p2)
        {
            if (p1.empty()) return true;
            if (p2.empty()) return false;
            return boost::lexical_cast<int>(p1) < boost::lexical_cast<int>(p2);
        }
        if (p1.empty() && p2.empty()) return false;
        b1 = e1;
        b2 = e2;
        if (b1 != std::string::npos) b1++;
        if (b2 != std::string::npos) b2++;
    }
    return false;
}

} // namespace mbody
} // namespace yimap
