#ifndef _YIMAP_BACKEND_META_COMMON_HELPERS_H_
#define _YIMAP_BACKEND_META_COMMON_HELPERS_H_

#include <macs/imap_folder.h>
#include <macs/label_set.h>
#include <macs/imap_envelope.h>
#include <yplatform/exception.h>

namespace yimap { namespace backend {

inline ::macs::ImapFolder convertFolder(const FolderInfo& folder)
{
    ::macs::ImapFolder result;
    result.revision = folder.revision;
    result.messageCount = folder.messageCount;
    result.recentCount = folder.recentCount;
    result.unseenCount = folder.unseenCount;
    result.uidNext = folder.uidNext;
    result.uidValidity = folder.uidValidity;
    result.name = folder.name;
    result.fid = folder.fid;
    return result;
}

inline void convertFlags(
    const macs::LabelSet& allLabels,
    const std::vector<std::string>& labels,
    Flags& flags)
{
    for (const string& lid : labels)
    {
        ::macs::LabelSet::const_iterator found = allLabels.find(lid);
        if (found == allLabels.end()) continue;

        const auto& label = found->second;
        if (label.symbolicName() == macs::Label::Symbol::seen_label)
        {
            flags.setFlag(Flags::MSG_SEEN);
        }
        else if (label.symbolicName() == macs::Label::Symbol::answered_label)
        {
            flags.setFlag(Flags::MSG_ANSWERED);
        }
        else if (label.symbolicName() == macs::Label::Symbol::deleted_label)
        {
            flags.setFlag(Flags::MSG_DELETED);
        }
        else if (label.symbolicName() == macs::Label::Symbol::draft_label)
        {
            flags.setFlag(Flags::MSG_DRAFT);
        }
        else if (label.symbolicName() == macs::Label::Symbol::recent_label)
        {
            flags.setFlag(Flags::MSG_RECENT);
        }
        else if (label.symbolicName() == macs::Label::Symbol::forwarded_label)
        {
            flags.setFlag(Flags::MSG_FORWARDED);
        }
        else if (label.symbolicName() == macs::Label::Symbol::important_label)
        {
            flags.setFlag(Flags::MSG_FLAGGED);
        }
        else
        {
            flags.setFlag(label.name());
        }
    }
}

inline std::list<macs::Label> createLabelsList(
    ::macs::LabelSet& allFlags,
    Flags flags,
    std::list<std::string>* newFlags)
{
    std::list<::macs::Label> result;
    for (const auto& flag : allFlags)
    {
        if (flags.hasFlag(Flags::MSG_SEEN) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::seen_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_ANSWERED) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::answered_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_DELETED) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::deleted_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_DRAFT) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::draft_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_RECENT) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::recent_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_FLAGGED) &&
            (flag.first == "priority_high" || flag.first == "FAKE_IMPORTANT_LBL"))
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_FLAGGED) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::important_label)
        {
            result.push_back(flag.second);
            continue;
        }
        if (flags.hasFlag(Flags::MSG_FORWARDED) &&
            flag.second.symbolicName() == ::macs::Label::Symbol::forwarded_label)
        {
            result.push_back(flag.second);
            continue;
        }

        if (flags.hasFlag(flag.second.name()))
        {
            result.push_back(flag.second);
        }
    }

    // Check flags, that do not exists in DB.
    flags.forEachFlag([allFlags, newFlags](const string& flag) {
        if (!newFlags) return;
        if (flag.empty() || flag[0] == '\\') return;

        if (!allFlags.exists(macs::LabelSet::Name(flag), macs::Label::Type::imap))
            newFlags->push_back(flag);
    });

    return result;
}

template <class Stream, class UidsContainer>
void logUids(Stream& stream, UidsContainer& uids)
{
    static const size_t LOG_UIDS_LIMIT_HALF = 5;
    size_t i = 0;
    stream << "[";
    for (auto uid : uids)
    {
        i++;
        if (i >= LOG_UIDS_LIMIT_HALF && (LOG_UIDS_LIMIT_HALF + i) < uids.size())
        {
            if (i == LOG_UIDS_LIMIT_HALF) stream << "... <total " << uids.size() << "> ... ";
            continue;
        }

        stream << uid << " ";
    }
    stream << "]";
}

template <typename Exception>
Future<void> createFuture(const Exception& e)
{
    Promise<void> prom;
    prom.set_exception(e);
    Future<void> result(prom);
    return result;
}

inline Future<void> createFuture(const string& /* publicError */, const string& privateError)
{
    Future<void> result = createFuture(std::runtime_error("yplatform::exception: " + privateError));
    return result;
}

inline Future<void> createFutureWithNoError()
{
    Promise<void> prom;
    prom.set();
    return Future<void>(prom);
}

}} // namespace

#endif //_YIMAP_BACKEND_META_COMMON_HELPERS_H_
