#pragma once

#include <backend/backend.h>
#include <common/folder.h>
#include <iostream>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

//-----------------------------------------------------------------------------
// Uid map block

class UidMapBlock
    : public UidMapEntry
    , public std::enable_shared_from_this<UidMapBlock>
{
public:
    typedef std::weak_ptr<UidMapBlock> WeakPtr;
    typedef std::shared_ptr<UidMapBlock> SharedPtr;

    mutable uint32_t num;
    uint32_t upperUid;

protected:
    WeakPtr prev;
    WeakPtr next;
    UidMapPtr messages;
    UidSet deletedUids;

public:
    // Helper message set to indicate block inconsistency. Temp solution
    UidMapPtr spoiled;

    UidMapBlock() : UidMapEntry(), num(0), upperUid(0)
    {
    }

    UidMapBlock(const UidMapEntry& uidMapEntry)
        : UidMapEntry(uidMapEntry)
        , num(0)
        , upperUid(0)
        , messages(new UidMap())
        , spoiled(new UidMap())
    {
    }

    void setPrev(UidMapBlock::SharedPtr prevBlock)
    {
        prev = prevBlock;
        SharedPtr prevPtr = prev.lock();
        if (prevPtr)
        {
            prevPtr->next = UidMapBlock::WeakPtr(shared_from_this());
            prevPtr->upperUid = uid - 1;
        }
        firstNum();
    }

    void insert(const MessageData& msg)
    {
        if (uid > msg.uid)
        {
            ostringstream errstr;
            errstr << "Inconsistent mailbox map in UidMapBlock::insert "
                   << "block.uid:" << uid << " msg.uid:" << msg.uid << ".";
            throw UidMapError("", errstr.str());
        }

        if (msg.uid > upperUid)
        {
            ostringstream errstr;
            errstr << "Inconsistent mailbox map in UidMapBlock::insert "
                   << "upperUid:" << upperUid << " msg.uid:" << msg.uid << ".";
            throw UidMapError("", errstr.str());
        }

        if (msg.added)
        {
            chain++;
        }
        MessageData message = msg;
        message.deleted = message.added = false;
        messages->insert(message);
    }

    bool isCached() const
    {
        return chain == 0 || chain == messages->size();
    }

    uint32_t firstNum() const
    {
        if (num != 0) return num;
        SharedPtr prevPtr = prev.lock();
        if (!prevPtr) return num = 1;
        num = prevPtr->firstNum() + prevPtr->chain;
        return num;
    }

    std::size_t size()
    {
        return messages->size();
    }

    static void messageFilter(
        const MessageData& msg,
        uint32_t index,
        uint32_t baseNum,
        const seq_range* seq,
        MessageData::Predicate pred,
        UidMap* dest)
    {
        MessageData message(msg);
        message.num = baseNum + index;
        if (seq->contains(message) && pred(message)) dest->insert(message);
    }

    void filterByRange(const seq_range& ranges, MessageData::Predicate pred, UidMap& result) const
    {
        uint32_t baseNum = firstNum();
        auto filter =
            std::bind(&UidMapBlock::messageFilter, ph::_1, ph::_2, baseNum, &ranges, pred, &result);
        messages->iterate(filter);
    }

    void filterByRange(const seq_range& ranges, size_t limit, UidMap& result) const
    {
        uint32_t baseNum = firstNum();
        auto i = 0;
        for (auto&& msg : *messages)
        {
            if (limit == 0) break;

            MessageData message(msg);
            message.num = baseNum + i;
            if (ranges.contains(message))
            {
                result.insert(message);
                --limit;
            }
            i++;
        }
    }

    bool canRemove(uint32_t uid) const
    {
        return deletedUids.find(uid) == deletedUids.end();
    }

    MessageData remove(const MessageData& msg)
    {
        if (uid > msg.uid)
        {
            ostringstream errstr;
            errstr << "Inconsistent mailbox map in UidMapBlock::remove "
                   << "block.uid:" << uid << " msg.uid:" << msg.uid << ".";
            throw UidMapError("", errstr.str());
        }

        if (msg.uid > upperUid)
        {
            ostringstream errstr;
            errstr << "Inconsistent mailbox map in UidMapBlock::remove "
                   << "upperUid:" << upperUid << " msg.uid:" << msg.uid << ".";
            throw UidMapError("", errstr.str());
        }

        MessageData victim(msg);
        // Message with nonzero baseUid already has correct number
        if (victim.baseUid == 0)
        {
            victim.num = firstNum() + (victim.offset > 0 ? victim.offset - 1 : 0);
        }
        messages->remove(msg.uid);
        deletedUids.insert(msg.uid);

        // chain == 0 is bad situation. We should remember bad message
        if (chain >= 1)
        {
            chain--;
        }
        else
        {
            spoiled->insert(victim);
        }
        invalidateNum();
        return victim;
    }

    std::string dump() const
    {
        std::ostringstream os;

        os << "BlockUid=" << uid << " "
           << "BlockNum=" << num << " "
           << "UpperUid=" << upperUid << std::endl;
        os << messages->dump() << std::endl;

        return os.str();
    }

    void updateMaxUid(uint32_t maxUid)
    {
        upperUid = maxUid;
    }
    range_t toUidRange() const
    {
        return range_t(uid, upperUid);
    }

    void updateBaseUid()
    {
        if (!messages->empty())
        {
            uid = messages->updateBaseUid();
        }
    }

protected:
    void invalidateNum()
    {
        SharedPtr n = shared_from_this();
        while (n && n->num != 0)
        {
            n->num = 0;
            n = n->next.lock();
        }
    }
};

typedef std::shared_ptr<UidMapBlock> UidMapBlockPtr;

//-----------------------------------------------------------------------------
// Blocked folder implementation

struct Comparator
{
    bool operator()(const UidMapBlock* f, const UidMapBlock* s)
    {
        if (f->upperUid < s->upperUid) return true;
        return false;
    }
};

// std::map<baseUid,block>
typedef std::map<uint32_t, UidMapBlockPtr> UidMapBlocks;
typedef std::set<const UidMapBlock*, Comparator> BlocksSelection;

class BlockedFolder
{
public:
    BlockedFolder(const UidMapData& uidMapData, const FolderInfo& aFolderInfo);

    void reset(const UidMapData& uidMapData);
    void change(const UidMapData& uidMapChanges);

    uint32_t changedNum(uint32_t baseUid, uint32_t offset)
    {
        if (blocks.empty()) return 1;
        return lowerByUid(baseUid)->second->firstNum() + offset;
    }

    BlocksSelection selectBlocks(const seq_range& ranges) const;
    void selectBlocks(const range_t& range, bool uidMode, BlocksSelection& result) const;
    void filterByRanges(const seq_range& ranges, MessageData::Predicate pred, UidMap& result) const;
    UidMap filterContinuouslyCachedByRanges(const seq_range& ranges, size_t limit) const;
    void filterPartialUids(const UidVector& uids, UidVector& result);

    std::tuple<size_t, seq_range> uncachedRanges(const seq_range& ranges) const;

    void insertMessage(const MessageData& msg);

    void updateMaxUid(uint32_t maxUid);
    uint32_t dropDeleted(const MessagesVector& deleted, MessagesVector& deletedResult);

    std::size_t size()
    {
        std::size_t ret = 0;
        for (auto&& [num, block] : blocks)
        {
            ret += block->size();
        }
        for (auto&& [num, block] : ghostBlocks)
        {
            ret += block->size();
        }
        return ret;
    }

    string dump() const;

    friend class MappedFolder;

protected:
    void linkList();
    void filterEmptyChains();
    void rebuildNumbers();
    void updateBaseUid(MessageVector& deleted);

    UidMapBlocks::iterator getBlock(size_t uid);
    UidMapBlocks::const_iterator lowerByUid(size_t uid) const;
    UidMapBlocks::const_iterator lowerByNum(size_t num) const;
    UidMapBlocks::const_iterator upperIter(UidMapBlocks::const_iterator lower) const;

    UidMapBlocks::iterator lastBlock()
    {
        return (++blocks.rbegin()).base();
    }
    UidMapBlocks::const_iterator lastBlock() const
    {
        return (++blocks.rbegin()).base();
    }

protected:
    const FolderInfo& finfo;
    UidMapBlocks blocks;
    UidMapBlocks ghostBlocks;
    UintMap numToUid;
};

inline BlockedFolder::BlockedFolder(const UidMapData& uidMapData, const FolderInfo& aFolderInfo)
    : finfo(aFolderInfo)
{
    reset(uidMapData);
}

inline void BlockedFolder::reset(const UidMapData& uidMapData)
{
    blocks.clear();
    numToUid.clear();

    for (const UidMapData::value_type& entryPair : uidMapData)
    {
        UidMapBlockPtr newBlock(new UidMapBlock(entryPair.second));
        blocks.insert(UidMapBlocks::value_type(entryPair.first, newBlock));
    }
    if (uidMapData.empty())
    {
        if (finfo.messageCount > 0)
            throw UidMapError(finfo.name, "Inconsistent mailbox map. Empty uidMapData.");

        UidMapEntry fakeEntry = { finfo.uidNext, 0, 0 };
        UidMapBlockPtr fakeBlock(new UidMapBlock(fakeEntry));
        blocks.insert(UidMapBlocks::value_type(fakeEntry.uid, fakeBlock));
    }
    linkList();
    lastBlock()->second->updateMaxUid(std::numeric_limits<uint32_t>::max());
    rebuildNumbers();
}

inline void BlockedFolder::change(const UidMapData& uidMapChanges)
{
    UidMapData newFullData;
    auto change = uidMapChanges.begin();
    auto block = blocks.begin();
    while (change != uidMapChanges.end() || block != blocks.end())
    {
        // For noew there are only new blocks. Add them to map
        if (block == blocks.end())
        {
            newFullData[change->second.uid] = change->second;
            change++;
            continue;
        }

        // There are no changes to process.
        // Fill rest of new data with current records.
        if (change == uidMapChanges.end())
        {
            newFullData[block->second->uid] = *block->second;
            block++;
            continue;
        }

        // Normaly this should never happened
        if (change->second.uid < block->second->uid)
        {
            change++;
            continue;
        }

        // Only change chain size for this block
        if (block->second->uid == change->second.uid)
        {
            newFullData[change->second.uid] = change->second;
            change++;
            continue;
        }

        // New block base uid inside old block
        if (change->second.uid <= block->second->upperUid)
        {
            // Ignore old block and use new one
            newFullData[change->second.uid] = change->second;
            change++;
            block++;
            continue;
        }
        // Changed block are further.
        newFullData[block->second->uid] = *block->second;
        block++;
    }

    reset(newFullData);
}

inline void BlockedFolder::linkList()
{
    UidMapBlockPtr prev;
    for (UidMapBlocks::value_type& block : blocks)
    {
        block.second->setPrev(prev);
        prev = block.second;
    }
}

inline void BlockedFolder::rebuildNumbers()
{
    numToUid.clear();
    for (const UidMapBlocks::value_type& block : blocks)
    {
        uint32_t num = block.second->firstNum();
        numToUid[num] = block.second->uid;
    }
}

inline void BlockedFolder::updateMaxUid(uint32_t maxUid)
{
    if (blocks.empty()) return;
    lastBlock()->second->updateMaxUid(maxUid);
}

inline void BlockedFolder::updateBaseUid(MessageVector& deleted)
{
    UidMapBlocks updateBlocks;

    for (auto& mess : deleted)
    {
        if (mess.uid == mess.baseUid)
        {
            auto block = blocks.find(mess.baseUid);
            if (block != blocks.end())
            {
                block->second->updateBaseUid();
                updateBlocks.insert(*block);
            }
        }
    }

    if (!updateBlocks.empty())
    {
        for (auto& block : updateBlocks)
        {
            blocks.erase(block.first);
            blocks.insert(UidMapBlocks::value_type(block.second->uid, block.second));
        }

        rebuildNumbers();
    }
}

inline UidMapBlocks::iterator BlockedFolder::getBlock(size_t uid)
{
    assert(!blocks.empty());
    UidMapBlocks::iterator lower = blocks.lower_bound(static_cast<uint32_t>(uid));
    if (lower == blocks.end()) lower = lastBlock();
    if (uid < lower->first && lower != blocks.begin()) lower--;
    return lower;
}

inline UidMapBlocks::const_iterator BlockedFolder::lowerByUid(size_t uid) const
{
    assert(!blocks.empty());
    UidMapBlocks::const_iterator lower = blocks.lower_bound(static_cast<uint32_t>(uid));
    if (lower == blocks.end()) lower = lastBlock();
    if (uid < lower->first && lower != blocks.begin()) lower--;
    return lower;
}

inline UidMapBlocks::const_iterator BlockedFolder::lowerByNum(size_t num) const
{
    assert(!numToUid.empty());
    UintMap::const_iterator lower = numToUid.lower_bound(static_cast<uint32_t>(num));
    if (lower == numToUid.end()) lower = (++numToUid.rbegin()).base();
    else if (num < lower->first && lower != numToUid.begin())
        lower--;

    UidMapBlocks::const_iterator result = blocks.find(lower->second);
    return result;
}

inline UidMapBlocks::const_iterator BlockedFolder::upperIter(
    UidMapBlocks::const_iterator lower) const
{
    return lower == blocks.end() ? lower : ++lower;
}

inline BlocksSelection BlockedFolder::selectBlocks(const seq_range& ranges) const
{
    BlocksSelection selection;
    for (const range_t& range : ranges)
    {
        selectBlocks(range, ranges.uidMode(), selection);
    }
    return selection;
}

inline void BlockedFolder::selectBlocks(const range_t& range, bool uidMode, BlocksSelection& result)
    const
{
    UidMapBlocks::const_iterator lower =
        uidMode ? lowerByUid(range.first) : lowerByNum(range.first);
    UidMapBlocks::const_iterator upper =
        upperIter(uidMode ? lowerByUid(range.second) : lowerByNum(range.second));

    for (UidMapBlocks::const_iterator blockIter = lower; blockIter != upper; blockIter++)
    {
        result.insert(blockIter->second.get());
    }
}

inline std::tuple<size_t, seq_range> BlockedFolder::uncachedRanges(const seq_range& ranges) const
{
    seq_range resultRanges(ranges.minVal(), ranges.maxVal(), true);
    BlocksSelection selectedBlocks = selectBlocks(ranges);

    auto it = selectedBlocks.begin();
    unsigned int rangeStart = 0;
    unsigned int rangeEnd = 0;
    unsigned int minimumSize = 0;
    while (it != selectedBlocks.end())
    {
        auto block = *it;
        if (block->isCached() && rangeStart)
        {
            resultRanges += range_t(rangeStart, rangeEnd);
            rangeStart = 0;
        }

        if (!block->isCached())
        {
            minimumSize = std::max(minimumSize, block->chain);
            if (!rangeStart)
            {
                rangeStart = block->uid;
            }
            rangeEnd = block->upperUid;
        }
        it++;
    }

    if (rangeStart) resultRanges += range_t(rangeStart, rangeEnd);
    return std::tuple{ minimumSize, resultRanges };
}

inline void BlockedFolder::filterByRanges(
    const seq_range& ranges,
    MessageData::Predicate pred,
    UidMap& result) const
{
    BlocksSelection selectedBlocks = selectBlocks(ranges);
    for (const UidMapBlock* block : selectedBlocks)
    {
        block->filterByRange(ranges, pred, result);
    }
}

inline UidMap BlockedFolder::filterContinuouslyCachedByRanges(const seq_range& ranges, size_t limit)
    const
{
    UidMap result;
    BlocksSelection selectedBlocks = selectBlocks(ranges);
    for (const UidMapBlock* block : selectedBlocks)
    {
        if (!block->isCached()) break;

        block->filterByRange(ranges, limit - result.size(), result);
    }
    return result;
}

inline void BlockedFolder::insertMessage(const MessageData& message)
{
    UidMapBlocks::iterator last = lastBlock();
    if (message.baseUid <= last->second->uid)
    {
        UidMapBlocks::iterator blockIter = getBlock(message.uid);
        if (blockIter->second->uid > message.uid)
        {
            // TODO: This is strange situation
            return;
        }
        blockIter->second->insert(message);
        return;
    }

    updateMaxUid(message.baseUid - 1);
    UidMapBlocks::iterator ghost = ghostBlocks.find(message.baseUid);
    if (ghost != ghostBlocks.end())
    {
        blocks.insert(*ghost);
        ghostBlocks.erase(ghost);
    }
    else
    {
        UidMapEntry newLastEntry = { message.baseUid, 0, 0 };
        UidMapBlockPtr newBlock(new UidMapBlock(newLastEntry));
        blocks.insert(UidMapBlocks::value_type(newLastEntry.uid, newBlock));
    }

    linkList();
    updateMaxUid(std::numeric_limits<uint32_t>::max());
    lastBlock()->second->insert(message);
    rebuildNumbers();
}

// Compute numbers numbers for deleted messages
inline uint32_t BlockedFolder::dropDeleted(
    const MessagesVector& deleted,
    MessagesVector& deletedResult)
{
    uint32_t maxRevision = finfo.revision;
    for (const MessageData& message : deleted)
    {
        maxRevision = std::max(maxRevision, message.modseq + 1);
        UidMapBlocks::iterator blockIter = getBlock(message.uid);
        if (blockIter->second->canRemove(message.uid))
        {
            MessageData victim = blockIter->second->remove(message);
            deletedResult.push_back(victim);
        }
    }
    filterEmptyChains();
    return maxRevision;
}

inline void BlockedFolder::filterEmptyChains()
{
    std::set<uint32_t> emptyChains;
    for (const UidMapBlocks::value_type& block : blocks)
    {
        if (block.second->chain == 0)
        {
            emptyChains.insert(block.second->uid);
        }
    }
    for (uint32_t baseUid : emptyChains)
    {
        if (blocks.size() <= 1) break;
        UidMapBlocks::iterator ghost = blocks.find(baseUid);
        if (ghost == blocks.end()) continue;
        ghostBlocks.insert(*ghost);
        blocks.erase(ghost);
    }
    linkList();
    updateMaxUid(std::numeric_limits<uint32_t>::max());
    rebuildNumbers();
}

inline void BlockedFolder::filterPartialUids(const UidVector& uids, UidVector& result)
{
    UidVector::const_iterator uid = uids.begin();
    UidVector::const_iterator end = uids.begin();
    while (uid != uids.end())
    {
        UidMapBlocks::iterator blockIter = getBlock(*uid);
        if (blockIter == blocks.end()) end = uids.end();
        else
            end = upper_bound(uid, uids.end(), blockIter->second->upperUid);
        result.insert(result.end(), uid, end);
        uid = end;
    }
}

inline std::string BlockedFolder::dump() const
{
    std::ostringstream os;

    os << "BLOCKS" << std::endl;
    for (auto block : blocks)
    {
        os << block.first << " " << std::endl;
        os << block.second->dump() << std::endl;
    }

    os << "NUM_TO_UID" << std::endl;
    for (auto& entry : numToUid)
    {
        os << "first=" << entry.first << " "
           << "second=" << entry.second << std::endl;
    }

    return os.str();
}

} // namespace backend
} // namespace yimap
