#include "blocked_folder.hpp"
#include <common/global_stats.h>

namespace yimap { namespace backend {

//-----------------------------------------------------------------------------
// MappedFolder use imap uid map to compute message numbers

class MappedFolder : public Folder
{
public:
    MappedFolder(const FolderInfo& aFolderInfo, const UidMapData& uidMapData, bool isShared)
        : Folder(aFolderInfo, isShared)
        , blocked(uidMapData, folderInfo)
        , lastKnownRevision(folderInfo.revision)
    {
    }

    virtual ~MappedFolder()
    {
        resetStats();
    }

    virtual size_t size() const;
    size_t knownRevision() const override;
    bool hasChanges() const override;

    // Mailbox modification.

    // messages array could be changed
    void update(MessagesVector& messages) override;
    void insertMessages(MessagesVector& messages) override;
    void filterPartialUids(const UidVector& uids, UidVector& result) override;

    MailboxDiffPtr updateToRevision(bool onlyChanged) override;

    // Selecting messages range
    UidMapPtr filterByRanges(const seq_range& ranges, MessageData::Predicate pred) const override;
    UidMap filterContinuouslyCachedByRanges(const seq_range& ranges, size_t limit) const override;
    std::tuple<size_t, seq_range> uncachedRanges(const seq_range& ranges) const override;

    void checkConsistency(Logger& logger) override;
    void debugDump(Logger& logger) override;
    string dump() const override;

protected:
    virtual void updateUidMap(const UidMapData& newUidMapData);
    virtual void postupdateUidMap()
    {
    }

    void updateRevision(uint32_t newRevision)
    {
        changedRevision = newRevision;
        folderInfo.revision = newRevision;
        lastKnownRevision = std::max(lastKnownRevision, newRevision);
    }

    void updateStats()
    {
        int64_t prevBlockedSize = blockedSize;
        blockedSize = blocked.size();
        GlobalStats::cachedMessagesCount += blockedSize - prevBlockedSize;
    }

    void resetStats()
    {
        GlobalStats::cachedMessagesCount -= blockedSize;
        blockedSize = 0;
    }

protected:
    BlockedFolder blocked;
    MessageMultiSet changes;
    MessageVector deleted;
    uint32_t lastKnownRevision;
    uint32_t changedRevision;
    int64_t blockedSize = 0; // Save size to prevent stats corruption

    mutable std::mutex fmutex;
};

size_t MappedFolder::size() const
{
    std_lock lock(fmutex);
    return folderInfo.messageCount;
}

size_t MappedFolder::knownRevision() const
{
    std_lock lock(fmutex);
    return lastKnownRevision;
}

bool MappedFolder::hasChanges() const
{
    std_lock lock(fmutex);
    return !changes.empty();
}

UidMapPtr MappedFolder::filterByRanges(const seq_range& ranges, MessageData::Predicate pred) const
{
    std_lock lock(fmutex);
    UidMapPtr result = std::make_shared<UidMap>();
    blocked.filterByRanges(ranges, pred, *result);
    return result;
}

UidMap MappedFolder::filterContinuouslyCachedByRanges(const seq_range& ranges, size_t limit) const
{
    std_lock lock(fmutex);
    return blocked.filterContinuouslyCachedByRanges(ranges, limit);
}

std::tuple<size_t, seq_range> MappedFolder::uncachedRanges(const seq_range& ranges) const
{
    std_lock lock(fmutex);
    return blocked.uncachedRanges(ranges);
}

void MappedFolder::update(MessagesVector& changedMessages)
{
    std_lock lock(fmutex);
    if (changedMessages.empty()) return;

    changedMessages.sortByRevision();

    UidMapData uidMapChanges;
    for (const MessageData& msg : changedMessages)
    {
        MessageData message = msg;
        lastKnownRevision = std::max(lastKnownRevision, message.modseq + 1);

        if (msg.uid == 0)
        {
            UidMapEntry entry = { msg.baseUid, msg.offset, 0 };
            uidMapChanges[entry.uid] = entry;
            continue;
        }

        message.added = message.uid >= folderInfo.uidNext;
        blocked.insertMessage(message);

        if (message.num == 0)
        {
            message.num = blocked.changedNum(message.baseUid, message.offset);
            if (msg.deleted) message.num = std::max(message.num - 1, 0u);
        }

        if (message.deleted)
        {
            if (message.modseq >= folderInfo.revision)
            {
                message.deleted = 0;
                if (message.added) changes.insert(message);
            }
            message.added = false;
            deleted.push_back(message);
        }
        else
        {
            changes.insert(message);
        }
    }

    updateUidMap(uidMapChanges);
    updateStats();
}

void MappedFolder::updateUidMap(const UidMapData& newUidMapData)
{
    if (!newUidMapData.empty()) blocked.reset(newUidMapData);
}

MailboxDiffPtr MappedFolder::updateToRevision(bool onlyChanged)
{
    std_lock lock(fmutex);

    uint32_t newRevision = folderInfo.revision;
    MessagesVector changed;
    MessagesVector added;
    MessagesVector deletedResult;

    FolderInfo folderInfoOriginal = folderInfo;
    if (!onlyChanged)
    {
        newRevision = blocked.dropDeleted(deleted, deletedResult);
        if (deletedResult.size() > folderInfo.messageCount)
        {
            folderInfo.messageCount = 0;
        }
        else
        {
            folderInfo.messageCount -= static_cast<uint32_t>(deletedResult.size());
        }
    }

    for (const MessageData& msg : changes)
    {
        if (msg.modseq + 1 < folderInfo.revision) continue;
        newRevision = std::max(newRevision, msg.modseq + 1);
        if (msg.added)
        {
            added.push_back(msg);
            folderInfo.messageCount = std::max(msg.num, folderInfo.messageCount);
            folderInfo.uidNext = std::max(folderInfo.uidNext, msg.uid + 1);
        }
        else
        {
            MessageData message(msg);
            changed.push_back(message);
        }
    }

    changes.clear();
    updateRevision(newRevision);
    postupdateUidMap();

    MailboxDiffPtr result;
    if (onlyChanged)
    {
        result.reset(new MailboxDiff(folderInfoOriginal, MessageVector(), changed, added));
    }
    else
    {
        result.reset(new MailboxDiff(folderInfoOriginal, deleted, changed, added));

        if (!deleted.empty())
        {
            blocked.updateBaseUid(deleted);
        }

        deleted.clear();
    }

    updateStats();

    return result;
}

void MappedFolder::insertMessages(MessagesVector& uids)
{
    std_lock lock(fmutex);
    for (const MessageData& msg : uids)
    {
        blocked.insertMessage(msg);
    }
    updateStats();
}

// From given uids vector, select only missing uids or uids without details
// uids must be sorted
void MappedFolder::filterPartialUids(const UidVector& uids, UidVector& result)
{
    std_lock lock(fmutex);
    blocked.filterPartialUids(uids, result);
}

void MappedFolder::debugDump(Logger& logger)
{
    std_lock lock(fmutex);
    logger.logDebug() << "MappedFolder dump " << folderInfo.name << ":" << folderInfo.fid
                      << " size:" << folderInfo.messageCount << " uidMap:" << blocked.dump();
}

string MappedFolder::dump() const
{
    std::stringstream ss;
    ss << "MappedFolder dump " << folderInfo.name << ":" << folderInfo.fid
       << " size:" << folderInfo.messageCount << " uidMap:" << blocked.dump();
    return ss.str();
}

void MappedFolder::checkConsistency(Logger& logger)
{
    std_lock lock(fmutex);
    ostringstream spoiledStream;
    bool spoiled = false;
    for (const UidMapBlocks::value_type& block : blocked.blocks)
    {
        if (block.second->spoiled->size())
        {
            spoiled = true;
            spoiledStream << "[block:" << block.second->uid << " uids: " << block.second->spoiled
                          << "] ";
        }
    }
    if (spoiled)
    {
        logger.logError() << "Mapped folder [" << folderInfo.name << ", fid:" << folderInfo.fid
                          << ", rev:" << folderInfo.revision << "]"
                          << " was spoiled: " << spoiledStream.str();
    }
}

//-----------------------------------------------------------------------------
// Little changes for postgres

class PgMappedFolder : public MappedFolder
{
public:
    PgMappedFolder(const FolderInfo& folderInfo, const UidMapData& uidMapData, bool isShared)
        : MappedFolder(folderInfo, uidMapData, isShared)
    {
    }

protected:
    // TODO: more coplex logic
    void updateUidMap(const UidMapData& newUidMapData) override
    {
        if (!newUidMapData.empty()) changedUidMapData = std::make_shared<UidMapData>(newUidMapData);
    }

    void postupdateUidMap() override
    {
        if (changedUidMapData && !changedUidMapData->empty()) blocked.change(*changedUidMapData);
    }

    std::shared_ptr<UidMapData> changedUidMapData;
};

//-----------------------------------------------------------------------------
// Folder factory methods

FolderPtr CreatePgMappedFolder(
    const FolderInfo& folderInfo,
    const UidMapData& uidMapData,
    bool isShared)
{
    FolderPtr folder(new PgMappedFolder(folderInfo, uidMapData, isShared));
    return folder;
}

} // namespace backend
} // namespace yimap
