#include "pg_backend.h"

#include "pg_changes.h"
#include "pg_copy_move.h"
#include "pg_expunge.h"
#include "pg_flags.h"
#include "pg_folder.h"
#include "pg_folder_info.h"
#include "pg_folders.h"
#include "pg_messages.h"
#include "pg_mimes.h"
#include "pg_settings.h"

#include <common/uid_map.h>
#include <backend/user_journal_types.h>

#include <macs/macs.h>

namespace yimap { namespace backend {

PgBackend::PgBackend(BackendServicePtr backendService, ImapContextPtr context)
    : backendService(backendService), context(context)
{
    if (context->settings->throttlingSettings)
    {
        throttlingController =
            std::make_shared<ThrottlingController>(*context->settings->throttlingSettings);
    }
}

Future<UserSettings> PgBackend::loadSettings()
{
    return worker<PgSettings>()->load();
}

Future<FolderListPtr> PgBackend::getFolders()
{
    return worker<PgFolders>()->load();
}

Future<FolderInfo> PgBackend::getFolderInfo(const DBFolderId& folderId)
{
    return worker<PgFolderInfo>()->loadFolderInfo(folderId);
}

Future<FolderPtr> PgBackend::getFolder(const DBFolderId& folderId)
{
    auto folderInfoWorker = worker<PgFolderInfo>();
    auto folderInfoFuture = folderInfoWorker->loadFolderInfo(folderId);
    return folderInfoFuture
        .then([=, capture_self](auto folderInfoFuture) {
            return folderInfoWorker->loadImapUidMap(folderInfoFuture.get());
        })
        .then([=, capture_self](auto uidmapFuture) {
            auto folderInfo = folderInfoFuture.get();
            bool isShared = context->foldersCache.getSharedStatusByFid(folderInfo.fid);
            return CreatePgMappedFolder(folderInfo, uidmapFuture.get(), isShared);
        });
}

Future<MailboxDiffPtr> PgBackend::statusUpdate(FolderRef mailbox)
{
    return worker<PgChanges>()->getChanges(mailbox).then([=, capture_self](auto future) {
        MailboxDiffPtr changes = future.get();
        if (changes && !changes->empty())
        {
            auto oldInfo = mailbox.info();
            return worker<PgFolderInfo>()
                ->loadFolderInfo(DBFolderId(oldInfo.name, oldInfo.fid))
                .then([=, capture_self](auto future) {
                    auto newInfo = future.get();
                    changes->setFolderInfo(newInfo);
                    return changes;
                });
        }
        else if (changes)
        {
            changes->setRecentChanged(false);
        }
        return makeFuture(changes);
    });
}

Future<UidMapPtr> PgBackend::loadMessages(FolderRef mailbox, const seq_range& range, bool partial)
{
    return runMessagesLoading([=, capture_self](uint64_t /*limit*/) {
        auto pgMessages = worker<PgMessages>();
        return worker<PgChanges>()
            ->getChanges(mailbox, true)
            .then([=, capture_self](auto future) mutable {
                future.get();
                return pgMessages->load(mailbox, range);
            })
            .then([=, capture_self](auto future) {
                future.get();
                if (partial) return makeFuture();
                return pgMessages->loadDetails(mailbox, range);
            })
            .then([=, capture_self](auto future) {
                future.get();
                auto res = mailbox.filterByRanges(range);
                updateStatsOnLoadMessages(res->size());
                return res;
            });
    });
}

Future<UidMapPtr> PgBackend::loadMessagesChunk(FolderRef mailbox, const seq_range& range)
{
    return runMessagesLoading([=, capture_self](uint64_t limit) {
        auto messages = mailbox.filterContinuouslyCachedByRanges(range, limit);
        if (messages.size())
        {
            updateStatsOnLoadMessages(messages.size());
            return makeFuture(std::make_shared<UidMap>(std::move(messages)));
        }

        auto pgMessages = worker<PgMessages>();
        return worker<PgChanges>()
            ->getChanges(mailbox, true)
            .then([=, capture_self](auto future) mutable {
                future.get();
                return pgMessages->loadChunk(mailbox, range, limit);
            })
            .then([=](auto future) {
                future.get();
                auto res = mailbox.filterContinuouslyCachedByRanges(range, limit);
                updateStatsOnLoadMessages(res.size());
                return std::make_shared<UidMap>(std::move(res));
            });
    });
}

Future<UidMapPtr> PgBackend::loadMessagesDetails(FolderRef mailbox, const seq_range& range)
{
    return runMessagesLoading([=, capture_self](uint64_t /*limit*/) {
        auto pgMessages = worker<PgMessages>();
        return pgMessages->loadDetails(mailbox, range).then([=, capture_self](auto future) {
            future.get();
            auto res = mailbox.filterByRanges(range);
            updateStatsOnLoadMessages(res->size());
            return res;
        });
    });
}

Future<UidMapPtr> PgBackend::loadMessagesToDelete(FolderRef mailbox, const seq_range& range)
{
    // TODO remove unnecessary messages loading for non empty range
    /* TODO refactor, empty range shouldn't mean "delete everything",
        e.g. "uid expunge 100500" deletes all messages if 100500 > uidmax
    */
    return runMessagesLoading([=, capture_self](uint64_t /*limit*/) {
        auto pgMessages = worker<PgDeletedMessages>();
        return pgMessages->load(mailbox.info()).then([=, capture_self](auto future) {
            UidMapPtr deletedUids = future.get();
            updateStatsOnLoadMessages(deletedUids->size());
            // We should load all blocks for deleting uids, because we should compute numbers
            seq_range targetRange = mailbox.seqRange(true);
            targetRange.insert(deletedUids->uidRange());

            return loadMessages(mailbox, range.empty() ? targetRange : range, true)
                .then([=, capture_self](auto future) {
                    auto messages = future.get();
                    set<string> deleted;
                    deleted.insert("\\Deleted");
                    return messages->filterByFlags(deleted, std::set<string>());
                });
        });
    });
}

Future<BodyMetadataByMid> PgBackend::loadBodyMetadata(const SmidList& mids)
{
    return worker<PgMimes>()->loadBodyMetadata(mids);
}

Future<UidMapPtr> PgBackend::getMessagesByMessageId(FolderRef mailbox, const string& messageId)
{
    return runMessagesLoading([=, capture_self](uint64_t /*limit*/) {
        auto pgMessages = worker<PgMessageId>();
        return pgMessages->loadByMessageId(mailbox.info(), messageId)
            .then([=, capture_self](auto future) {
                auto res = future.get();
                updateStatsOnLoadMessages(res->size());
                return res;
            });
    });
}

Future<MailboxDiffPtr> PgBackend::updateFlags(
    FolderRef mailbox,
    UidMapPtr uids,
    const Flags& del_flags,
    const Flags& add_flags)
{
    return worker<PgFlags>()->update(mailbox, uids, del_flags, add_flags);
}

Future<void> PgBackend::expunge(FolderRef mailbox, UidMapPtr uids)
{
    auto expungeWorker = worker<PgExpunge>();
    // XXX refactor workers creation to avoid dirty initialization
    expungeWorker->folder = mailbox;
    expungeWorker->messages = uids;
    yplatform::spawn(expungeWorker);
    return expungeWorker->promise;
}

Future<UidSequence> PgBackend::copyMessages(
    UidMap& uids,
    const FolderInfo& from,
    const FolderInfo& to)
{
    auto future = worker<PgCopyMove>()->copy(uids, from, to);
    return future
        .then([=, capture_self](auto future) {
            auto revision = future.get();
            return getChangesOnRevision(to, revision);
        })
        .then([capture_self](auto future) { return future.get(); });
}

Future<UidSequence> PgBackend::moveMessages(
    UidMap& uids,
    const FolderInfo& from,
    const FolderInfo& to)
{
    auto future = worker<PgCopyMove>()->move(uids, from, to);
    return future
        .then([=, capture_self](auto future) {
            auto revision = future.get();
            return getChangesOnRevision(to, revision);
        })
        .then([capture_self](auto future) { return future.get(); });
}

Future<UidSequence> PgBackend::getChangesOnRevision(const FolderInfo& folder, unsigned revision)
{
    return worker<PgChanges>()->getChanges(folder).then([revision](auto future) -> UidSequence {
        UidSequence res;
        auto changes = future.get();
        for (auto&& msg : *changes)
        {
            if (msg.modseq == revision) res.push(msg.uid);
        }
        return res;
    });
}

Future<void> PgBackend::createFolder(const string& folderName)
{
    return worker<PgFolder>()->create(folderName);
}

Future<void> PgBackend::deleteFolder(const DBFolderId& folder)
{
    return worker<PgFolder>()->remove(folder);
}

Future<void> PgBackend::renameFolder(const DBFolderId& folder, const string& destName)
{
    return worker<PgFolder>()->rename(folder, destName);
}

Future<void> PgBackend::subscribe(const DBFolderId& folderId, bool on)
{
    return worker<PgFolder>()->subscribe(folderId, on);
}

Future<void> PgBackend::dropFreshCounter()
{
    return worker<PgEnvelopes>()->resetFreshCounter();
}

Future<string> PgBackend::regenerateImapId(const std::string& mid)
{
    return worker<PgMessages>()->regenerateImapId(mid);
}

void PgBackend::journalAuth()
{
    auto journal = backendService->createJournal(
        context->userData.uid,
        context->sessionInfo.remoteAddress,
        context->userData.storage,
        context->userData.suid);

    if (journal)
    {
        journal->logOperation<user_journal_types::Authorization>(
            user_journal::parameters::id::state(std::string()),
            user_journal::parameters::id::affected(0u));
    }
}

PgServicePtr PgBackend::createPgService(bool replica)
{
    auto sessInfo = createShortSessionInfo(*context);
    std_lock lock(pgMutex);
    PgServicePtr& pgService = replica ? pgReplica : pgMaster;
    if (!pgService)
    {
        pgService = backendService->createPgService(sessInfo, replica);
    }
    return pgService;
}

} // namespace backend
} // namespace yimap
