#pragma once

#include <backend/backend_service.h>
#include <backend/throttling_controller.h>
#include <common/imap_context.h>
#include <macs_pg/macs_pg.h>
#include <common/settings.h>
#include <common/messages_stats.h>

namespace yimap { namespace backend {

class PgBackend
    : public MetaBackend
    , public UserSettingsBackend
    , public std::enable_shared_from_this<PgBackend>
{
public:
    PgBackend(BackendServicePtr backendService, ImapContextPtr context);
    virtual ~PgBackend()
    {
    }

    Future<UserSettings> loadSettings() override;

    Future<FolderListPtr> getFolders() override;

    virtual Future<FolderPtr> getFolder(const DBFolderId& folderId);
    virtual Future<FolderInfo> getFolderInfo(const DBFolderId& folderId);

    // Loading messages
    virtual Future<UidMapPtr> loadMessages(FolderRef mailbox, const seq_range& range, bool partial);
    virtual Future<UidMapPtr> loadMessagesChunk(FolderRef mailbox, const seq_range& range);
    virtual Future<UidMapPtr> loadMessagesDetails(FolderRef mailbox, const seq_range& range);
    virtual Future<UidMapPtr> loadMessagesToDelete(FolderRef mailbox, const seq_range& range);
    virtual Future<UidMapPtr> getMessagesByMessageId(FolderRef mailbox, const string& messageId);
    virtual Future<BodyMetadataByMid> loadBodyMetadata(const SmidList& mids);

    virtual Future<MailboxDiffPtr> statusUpdate(FolderRef mailbox);

    // Working with message flags
    virtual Future<MailboxDiffPtr> updateFlags(
        FolderRef mailbox,
        UidMapPtr uids,
        const Flags& del_flags,
        const Flags& add_flags);

    virtual Future<void> expunge(FolderRef mailbox, UidMapPtr uids);

    virtual Future<UidSequence> copyMessages(
        UidMap& uids,
        const FolderInfo& from,
        const FolderInfo& to);
    virtual Future<UidSequence> moveMessages(
        UidMap& uids,
        const FolderInfo& from,
        const FolderInfo& to);

    virtual Future<void> createFolder(const string& folderName);
    virtual Future<void> deleteFolder(const DBFolderId& folderId);
    virtual Future<void> renameFolder(const DBFolderId& folderId, const string& dstName);
    virtual Future<void> subscribe(const DBFolderId& folderId, bool on);

    virtual Future<void> dropFreshCounter();
    virtual void journalAuth();

    virtual Future<string> regenerateImapId(const std::string& mid);

    ImapContextPtr getContext()
    {
        return context;
    }

    PgServicePtr createPgService(bool replica);

    void updateStatsOnLoadMessages(size_t count)
    {
        context->messagesStats.loadedFromMetabackendCount += count;
    }

protected:
    Future<UidSequence> getChangesOnRevision(const FolderInfo& folder, unsigned revision);

    template <typename TypeWorker>
    std::shared_ptr<TypeWorker> worker()
    {
        return std::make_shared<TypeWorker>(*this);
    }

    template <typename LoadMessages>
    auto runMessagesLoading(LoadMessages&& loadMessagesImpl)
    {
        auto fetchChunkSize = context->settings->fetchChunkSize;
        if (!throttlingController)
        {
            return loadMessagesImpl(fetchChunkSize);
        }

        auto self = shared_from_this();
        auto loadImplWrapper = [impl = std::move(loadMessagesImpl), fetchChunkSize, self, this](
                                   size_t throttlingLimit) {
            auto chunkLimit = std::min(throttlingLimit, fetchChunkSize);
            return impl(chunkLimit).then([self, this](auto futureUidMapPtr) {
                auto res = futureUidMapPtr.get();
                throttlingController->consume(res->size());
                return res;
            });
        };

        if (auto currentLimit = throttlingController->limit())
        {
            return loadImplWrapper(currentLimit);
        }

        return wait(throttlingController->recoveryDelay())
            .then([loadImplWrapper, self, this](auto future) {
                future.get();
                return loadImplWrapper(throttlingController->limit());
            });
    }

    Future<void> wait(Duration delay)
    {
        Promise<void> prom;
        auto timer = createTimer(context, delay);
        timer->async_wait([timer, prom](auto ec) mutable {
            if (ec != boost::asio::error::operation_aborted)
            {
                prom.set();
            }
        });
        return prom;
    }

    BackendServicePtr backendService;
    ImapContextPtr context;
    ThrottlingControllerPtr throttlingController = nullptr;
    std::mutex pgMutex;
    PgServicePtr pgMaster;
    PgServicePtr pgReplica;
};

} // namespace backend
} // namespace yimap
