#ifndef _YIMAP_PROCESSOR_CACHE_MACS_CHANGES_H_
#define _YIMAP_PROCESSOR_CACHE_MACS_CHANGES_H_

#include <backend/meta_pg/pg_worker.h>

#include <macs/imap_repository.h>
#include <macs/hooks.h>
#include <macs/imap_envelope.h>
#include <backend/meta_common/helpers.h>

#include <boost/bind.hpp>
#include <boost/function.hpp>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

class PgChanges : public PgWorker
{
public:
    PgChanges(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<MessagesVectorPtr> getChanges(const FolderInfo& folder)
    {
        return loadImapFlags().then([=, capture_self](auto future) {
            Promise<MessagesVectorPtr> promise;
            auto imapFlags = future.get();
            imapRepo().imapGetChanges(
                convertFolder(folder),
                [self, imapFlags, logger = &logger, promise](auto err, auto changes) mutable {
                    if (err)
                    {
                        logger->logError() << "Error from getChanges#1: " << err.full_message();
                        promise.set_exception(BackendError(err.full_message()));
                        return;
                    }
                    promise.set(convertPgChanges(changes, imapFlags));
                });
            return Future<MessagesVectorPtr>(promise);
        });
    }

    Future<MailboxDiffPtr> getChanges(FolderRef mailbox, bool withoutUpdate = false)
    {
        return loadImapFlags()
            .then([=, capture_self](auto future) mutable {
                auto imapFlags = future.get();

                auto folder = convertFolder(mailbox.info());

                // If maibox contains changes, with greater revision, skip it
                folder.revision = mailbox.knownRevision();

                auto logHelper = std::make_shared<LogHelper>(logger.logDebug());
                (*logHelper) << "Loading changes for \"" << folder.name << "\""
                             << " uidvalidity=" << folder.uidValidity << " rev=" << folder.revision
                             << ".";

                Promise<macs::ImapEnvelopeChangesChunk> promise;
                imapRepo().imapGetChanges(
                    folder, [=, capture_self](auto err, auto changes) mutable {
                        if (err)
                        {
                            logger.logError() << "Error from getChanges#2: " << err.full_message();
                            promise.set_exception(BackendError(err.full_message()));
                        }
                        else
                        {
                            promise.set(changes);
                        }
                    });
                return Future<macs::ImapEnvelopeChangesChunk>(promise).then(
                    [=, capture_self](auto&& future) mutable {
                        auto changes = future.get();
                        mailbox.update(*convertPgChanges(changes, imapFlags));
                        mailbox.checkConsistency(logger);
                        (*logHelper) << " Got " << changes.size() << " changed messages";
                        if (withoutUpdate) return MailboxDiffPtr();
                        return mailbox.updateToRevision();
                    });
            })
            .then([](auto future) { return future.get(); });
    }
};
}} // namespace

#endif // _YIMAP_PROCESSOR_CACHE_MACS_CHANGES_H_
