#ifndef _YIMAP_PROCESSOR_CACHE_MACS_COPY_H_
#define _YIMAP_PROCESSOR_CACHE_MACS_COPY_H_

#include <backend/meta_pg/pg_worker.h>

#include <macs/hooks.h>
#include <macs/envelopes_repository.h>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

class PgCopyMove : public PgWorker
{
public:
    PgCopyMove(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<Revision> copy(UidMap& messages, const FolderInfo& from, const FolderInfo& to)
    {
        auto operation = [=, capture_self](auto&&... args) mutable {
            envelopesRepo().copyMessages(args...);
        };
        using Mids = std::vector<string>;
        return perform<decltype(operation), Mids, CopyError>(operation, messages, from, to);
    }

    Future<Revision> move(UidMap& messages, const FolderInfo& from, const FolderInfo& to)
    {
        auto operation = [=, capture_self](auto&&... args) mutable {
            envelopesRepo().moveMessages(args...);
        };
        using Mids = std::list<string>;
        return perform<decltype(operation), Mids, MoveError>(operation, messages, from, to);
    }

private:
    template <typename Operation, typename Mids, typename Error>
    Future<Revision> perform(
        Operation operation,
        const UidMap& messages,
        const FolderInfo& from,
        const FolderInfo& to)
    {
        Promise<Revision> promise;

        auto mids = std::make_shared<Mids>();
        messages.copyMids(*mids);
        auto chunker = std::make_shared<Chunker<Mids>>(mids, settings->maxDBChunkSize);

        auto runOp = [=, capture_self](auto rev, auto continuation) mutable {
            if (chunker->empty()) return promise.set(rev);
            auto chunk = chunker->next();
            operation(to.fid, chunk, [=, capture_self](auto err, auto result) mutable {
                if (err)
                {
                    promise.set_exception(Error(from.name, to.name, err.message()));
                    return;
                }
                auto revision = getRevision(result) - 1;
                continuation(revision, continuation);
            });
        };
        runOp(0, runOp);

        return promise;
    }

    template <typename Result>
    static Revision getRevision(Result result)
    {
        if constexpr (std::is_same_v<macs::Revision, Result>)
        {
            return result;
        }
        else
        {
            return result.rev;
        }
    }
};

}} // namespace

#endif // _YIMAP_PROCESSOR_CACHE_MACS_COPY_H_
