#ifndef _YIMAP_BACKEND_META_PG_PG_FLAGS_H_
#define _YIMAP_BACKEND_META_PG_PG_FLAGS_H_

#include <common/folder.h>
#include <common/uid_map.h>

#include <backend/meta_common/helpers.h>
#include <backend/meta_pg/pg_worker.h>
#include <macs/envelopes_repository.h>
#include <macs/labels_repository.h>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

class PgFlags : public PgWorker
{
public:
    PgFlags(PgBackend& backend) : PgWorker(backend), destModseq(0)
    {
    }

    Future<MailboxDiffPtr> update(
        FolderRef folder,
        UidMapPtr messages,
        const Flags& delFlags,
        const Flags& addFlags)
    {
        mailbox = folder;
        addFlags_ = addFlags;
        delFlags_ = delFlags;
        return loadImapFlags()
            .then([this, capture_self, messages](auto&& future) {
                auto allLabels = future.get();
                delLabels_ = createLabelsList(allLabels, delFlags_, nullptr);
                addLabels_ = createLabelsList(allLabels, addFlags_, &newLabelNames_);
                return createNewLabels();
            })
            .then([this, capture_self, messages](auto&& future) {
                future.get();
                addLabels_.insert(addLabels_.end(), newLabels_.begin(), newLabels_.end());
                pgMaster().labels().resetLabelsCache();
                pgReplica().labels().resetLabelsCache();
                return markEnvelopes(messages->toSmidList(), addLabels_);
            })
            .then([this, capture_self, messages](auto&& future) {
                future.get();
                return unmarkEnvelopes(messages->toSmidList(), delLabels_);
            })
            .then([this, capture_self, messages](auto&& future) {
                future.get();
                return createDiff(messages);
            });
    }

    Future<void> markEnvelopes(SmidListPtr midList, std::list<macs::Label>& addLabels)
    {
        if (addLabels.empty()) return makeFuture();
        auto chunker = std::make_shared<Chunker<SmidList>>(midList, settings->maxDBChunkSize);
        return applyToChunks(chunker, [this, capture_self, addLabels](auto&& mids) -> Future<void> {
            Promise<void> promise;
            pg->envelopes().markEnvelopes(
                mids,
                addLabels,
                [this, capture_self, promise](auto&& err, auto&& revision) mutable {
                    if (err)
                    {
                        return promise.set_exception(BackendError(err.full_message()));
                    }
                    if (revision > destModseq) destModseq = revision - 1;
                    promise.set();
                });
            return promise;
        });
    }

    Future<void> unmarkEnvelopes(SmidListPtr midList, std::list<macs::Label>& delLabels)
    {
        if (delLabels.empty()) return makeFuture();
        auto chunker = std::make_shared<Chunker<SmidList>>(midList, settings->maxDBChunkSize);
        return applyToChunks(chunker, [this, capture_self, delLabels](auto&& mids) -> Future<void> {
            Promise<void> promise;
            pg->envelopes().unmarkEnvelopes(
                mids,
                delLabels,
                [this, capture_self, promise](auto&& err, auto&& revision) mutable {
                    if (err)
                    {
                        return promise.set_exception(BackendError(err.full_message()));
                    }
                    if (revision > destModseq) destModseq = revision - 1;
                    promise.set();
                });
            return promise;
        });
    }

    Future<void> createNewLabels()
    {
        if (newLabelNames_.empty()) return makeFuture();
        return applyToRange(
            newLabelNames_.begin(),
            newLabelNames_.end(),
            [this, capture_self](auto&& labelName) mutable -> Future<void> {
                Promise<void> promise;
                labelsRepo().createLabelWithType(
                    labelName,
                    "",
                    "imap",
                    [this, capture_self, promise](auto&& err, auto&& label) mutable {
                        if (err)
                        {
                            return promise.set_exception(BackendError(err.full_message()));
                        }
                        newLabels_.emplace_back(label);
                        promise.set();
                    });
                return promise;
            });
    }

protected:
    MailboxDiffPtr createDiff(UidMapPtr messages)
    {
        MessageVector changed;

        while (!messages->empty())
        {
            auto message = messages->pop();
            message.flags.delFlags(delFlags_);
            message.flags.addFlags(addFlags_);
            changed.push_back(message);
        }

        // If nothing was changed
        if (destModseq == 0)
        {
            // We should return current messages state
            return std::make_shared<MailboxDiff>(
                mailbox.info(), MessageVector(), changed, MessageVector());
        }

        for (auto& message : changed)
        {
            message.modseq = destModseq;
        }
        // And update local cache
        mailbox.update(changed);
        return mailbox.updateToRevision(true);
    }

private:
    Flags delFlags_;
    Flags addFlags_;
    FolderRef mailbox;

    std::list<macs::Label> addLabels_;
    std::list<macs::Label> delLabels_;

    std::list<string> newLabelNames_;
    std::list<macs::Label> newLabels_;

    uint32_t destModseq = 0;
};

} // namespace backend
} // namespace yimap

#endif // _YIMAP_BACKEND_META_PG_PG_FLAGS_H_
