#ifndef _YIMAP_BACKEND_META_PG_PG_FOLDER_H_
#define _YIMAP_BACKEND_META_PG_PG_FOLDER_H_

#include <backend/meta_common/helpers.h>
#include <backend/meta_pg/pg_worker.h>
#include <macs/folders_repository.h>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

class PgFolder : public PgWorker
{
public:
    PgFolder(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<void> create(const string& folderName)
    {
        Promise<void> promise;
        string errorMessage = "Failed to create folder " + folderName;

        if (folderName.length() > macs::Folder::maxFolderNameLength())
        {
            promise.set_exception(
                FolderNameTooLargeError(folderName, errorMessage + ". Folder name too large"));
            return promise;
        }

        vector<string> parts;
        boost::split(parts, folderName, boost::is_any_of("|"));

        foldersRepo().createFolderByPath(
            macs::Folder::Path(parts), [=, capture_self](auto err, auto /*folder*/) mutable {
                if (err)
                {
                    promise.set_exception(BackendError(err.full_message()));
                    return;
                }
                promise.set();
            });

        return promise;
    }

    Future<void> rename(const DBFolderId& folderId, const string& dstName)
    {
        Promise<void> promise;

        string errorMessage =
            "Failed to rename folder from \"" + folderId.name + "\"" + " to \"" + dstName + "\"";
        if (dstName.size() > macs::Folder::maxFolderNameLength())
        {
            promise.set_exception(
                FolderNameTooLargeError(dstName, errorMessage + ". Folder name is too large"));
            return promise;
        }
        macs::Folder folder = foldersRepo().getFolderByFid(folderId.fid);
        macs::FolderFactory factory(std::move(folder));
        macs::Folder::Path path(dstName);

        foldersRepo().updateFolderToPath(
            std::move(factory),
            std::move(path),
            [=, capture_self](auto err, auto /*folder*/) mutable {
                if (err)
                {
                    promise.set_exception(BackendError(err.full_message()));
                    return;
                }
                promise.set();
            });

        return promise;
    }

    Future<void> remove(const DBFolderId& folderId)
    {
        Promise<void> promise;
        auto folder = foldersRepo().getFolderByFid(folderId.fid);
        foldersRepo().eraseCascade(folder, [=, capture_self](auto err, auto /*folder*/) mutable {
            if (err)
            {
                promise.set_exception(BackendError(err.full_message()));
                return;
            }
            promise.set();
        });
        return promise;
    }

    Future<void> subscribe(const DBFolderId& folder, bool on)
    {
        std::vector<std::string> folderNameParts;
        boost::split(folderNameParts, folder.name, boost::is_any_of("|"));

        Promise<void> promise;
        auto subscribeHandler =
            [=, capture_self](const macs::sys::error_code& ec, macs::Revision) mutable {
                if (ec)
                {
                    promise.set_exception(BackendError(ec.message()));
                }
                else
                {
                    promise.set();
                }
            };

        if (on)
        {
            imapRepo().imapSubscribeFolder(folderNameParts, subscribeHandler);
        }
        else
        {
            imapRepo().imapUnsubscribeFolder(folderNameParts, subscribeHandler);
        }

        return promise;
    }
};

} // namespace backend
} // namespace yimap

#endif // _YIMAP_BACKEND_META_PG_PG_FOLDER_H_
