#ifndef _YIMAP_BACKEND_META_PG_PG_FOLDER_INFO_H_
#define _YIMAP_BACKEND_META_PG_PG_FOLDER_INFO_H_

#include <macs/hooks.h>
#include <macs/folders_repository.h>

namespace yimap { namespace backend {

namespace ph = std::placeholders;

class PgFolderInfo : public PgWorker
{
public:
    PgFolderInfo(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<FolderInfo> loadFolderInfo(const DBFolderId& folderId)
    {
        Promise<FolderInfo> promise;
        auto& pgFolders = foldersRepo();
        pgFolders.resetFoldersCache();

        pgFolders.getAllFolders([=, capture_self](auto ec, auto&& folders) mutable {
            if (ec)
            {
                return promise.set_exception(FolderListError(ec.message()));
            }
            auto it = folders.find(folderId.fid);
            if (it == folders.end())
            {
                return promise.set_exception(
                    NoSuchFolderError(folderId.name, "fid:" + folderId.fid));
            }
            promise.set(toFolderInfo(it->second, folders.getPath(it->second)));
        });
        return promise;
    }

    Future<UidMapData> loadImapUidMap(const FolderInfo& folderInfo)
    {
        Promise<UidMapData> promise;
        auto macsFolder = toMacsFolder(folderInfo);
        updateTargetRevision(folderInfo.revision);

        auto&& repo = imapRepo(ReplicaState::TryReplica);
        repo.imapGetUidMap(
            macsFolder, [=, capture_self, name = macsFolder.name](auto ec, auto&& chunk) mutable {
                if (ec)
                {
                    return promise.set_exception(UidMapError(name, ec.message()));
                }
                UidMapData result;
                for (auto&& i : chunk)
                {
                    result[i.uid] = UidMapEntry{ i.uid, i.chain, 0 };
                }
                promise.set(result);
            });
        return promise;
    }

protected:
    macs::ImapFolder toMacsFolder(const FolderInfo& folderInfo)
    {
        macs::ImapFolder imapFolder;
        imapFolder.name = folderInfo.name;
        imapFolder.fid = folderInfo.fid;
        imapFolder.revision = folderInfo.revision;
        return imapFolder;
    }

    FolderInfo toFolderInfo(const ::macs::Folder& folder, const ::macs::Folder::Path& path)
    {
        FolderInfo info;
        info.name = path.toString();
        info.fid = folder.fid();
        info.revision = static_cast<uint32_t>(folder.revision());
        info.messageCount = static_cast<uint32_t>(folder.messagesCount());
        info.recentCount = static_cast<uint32_t>(folder.recentMessagesCount());
        info.unseenCount = static_cast<uint32_t>(folder.newMessagesCount());
        info.uidNext = static_cast<uint32_t>(folder.imapUidNext() > 0 ? folder.imapUidNext() : 0);
        info.uidValidity = static_cast<uint32_t>(folder.imapUidValidity());
        info.firstUnseen = static_cast<uint32_t>(folder.imapFirstUnseen());
        // info.sysdate      = static_cast<uint32_t>(folder.sysdate);
        // info.changed      = static_cast<uint32_t>(folder.changed);
        // info.mapChanged   = static_cast<uint32_t>(folder.mapChanged);
        return info;
    }
};

}} // namespace

#endif // _YIMAP_BACKEND_META_PG_PG_FOLDER_INFO_H_
