#ifndef _YIMAP_BACKEND_META_PG_PG_FOLDERS_H_
#define _YIMAP_BACKEND_META_PG_PG_FOLDERS_H_

#include <common/folder_list.h>

#include <macs/folders_repository.h>
#include <macs_pg/macs_pg.h>

#include <iostream>

namespace yimap { namespace backend {

class PgFolders : public PgWorker
{
public:
    PgFolders(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<FolderListPtr> load()
    {
        Promise<macs::FolderSet> promise;
        auto& pgFolders = foldersRepo();
        pgFolders.resetFoldersCache();
        pgFolders.getAllFolders([=, capture_self](auto&& err, auto&& folders) mutable {
            if (err)
            {
                promise.set_exception(FolderListError(err.full_message()));
            }
            else
            {
                promise.set(folders);
            }
        });
        return Future<macs::FolderSet>(promise).then([=, capture_self](auto&& future) mutable {
            auto folders = future.get();
            return loadUnsubscribed().then([=, capture_self](auto&& future) mutable {
                auto unsubscribed = future.get();
                return makeResultFolders(folders, unsubscribed);
            });
        });
    };

protected:
    using UnsubscribedSet = std::set<std::string>;

    FolderListPtr makeResultFolders(
        const ::macs::FolderSet& folders,
        const UnsubscribedSet& unsubscribed)
    {
        auto rawFolders = std::make_shared<RawFolderList>();

        for (::macs::FolderSet::const_iterator folderIter = folders.begin();
             folderIter != folders.end();
             folderIter++)
        {
            const macs::Folder& folder = folderIter->second;

            FullFolderInfoPtr new_folder(new FullFolderInfo());

            new_folder->name = folders.getPath(folder).toString();
            new_folder->fid = folder.fid();
            new_folder->symbol = folder.symbolicName().title();

            new_folder->recent = false;
            new_folder->isShared = folder.subscribedForSharedFolder();

            new_folder->messageCount = folder.messagesCount();
            new_folder->unseenCount = folder.newMessagesCount();
            new_folder->recentCount = folder.recentMessagesCount();

            // Folder is subscribed if its name is not in unsubscribed set
            new_folder->subscribed = unsubscribed.count(new_folder->name) == 0;

            rawFolders->push_back(new_folder);
        }

        auto language = std::make_shared<LanguageConfig>(
            context->settings->languageOptionsXml,
            *context->userSettings,
            context->userData.language);
        return std::make_shared<FolderList>(rawFolders, language);
    }

    Future<UnsubscribedSet> loadUnsubscribed()
    {
        Promise<UnsubscribedSet> promise;
        imapRepo().imapGetUnsubscribed([=, capture_self](auto&& ec, auto&& chunk) mutable {
            if (ec)
            {
                promise.set_exception(FolderListError(ec.full_message()));
                return;
            }
            UnsubscribedSet unsubscribed;
            for (auto&& unsubscribedFolder : chunk)
            {
                unsubscribed.insert(boost::join(unsubscribedFolder.fullName, "|"));
            }
            promise.set(unsubscribed);
        });
        return promise;
    }
};

}} // namespace

#endif //_YIMAP_BACKEND_META_PG_PG_FOLDERS_H_
