#include "pg_messages.h"
#include <backend/meta_common/helpers.h>

#include <macs/envelopes_repository.h>
#include <macs/labels_repository.h>
#include <macs/label.h>

namespace yimap { namespace backend {

using namespace ::macs;
namespace ph = std::placeholders;

PgMessages::PgMessages(PgBackend& backend) : PgWorker(backend), context(backend.getContext())
{
}

Future<void> PgMessages::load(FolderRef mailbox, const seq_range& range)
{
    auto [minimumSize, rangeToCache] = mailbox.uncachedRanges(range);
    auto mailboxInfo = mailbox.info();
    updateTargetRevision(mailboxInfo.revision);

    if (range.empty()) return makeFuture();

    return loadImapFlags(ReplicaState::TryReplica)
        .then([=, rangeToCache = rangeToCache, capture_self](auto future) mutable {
            imapFlags = future.get();
            return loadMessages(mailboxInfo, rangeToCache, [this](auto&&... args) {
                imapRepo(ReplicaState::TryReplica).imapGetMails(args...);
            });
        })
        .then([=, capture_self](auto future) mutable {
            future.get();
            mailbox.insertMessages(messages);
        });
}

Future<void> PgMessages::loadChunk(FolderRef mailbox, const seq_range& range, size_t limit)
{
    auto [minimumSize, rangesToCache] = mailbox.uncachedRanges(range);
    limit = std::max(limit, minimumSize);
    auto mailboxInfo = mailbox.info();
    updateTargetRevision(mailboxInfo.revision);

    if (rangesToCache.empty()) return makeFuture();

    return loadImapFlags(ReplicaState::TryReplica)
        .then([=, range = takeChunk(rangesToCache), capture_self](auto future) mutable {
            imapFlags = future.get();
            return loadMessages(
                mailboxInfo,
                range,
                [=](const ImapFolder& folder,
                    uint64_t low,
                    uint64_t hi,
                    ImapRepository::ImapListMode /*mode*/,
                    auto handler) {
                    imapRepo(ReplicaState::TryReplica)
                        .imapGetMailsChunk(folder, low, hi, limit, handler);
                });
        })
        .then([=, capture_self](auto future) mutable {
            future.get();
            mailbox.insertMessages(messages);
        });
}

void PgMessages::convertMessages(const ImapEnvelopeChunk& messagesChunk, MessageVector& destination)
{
    for (const ImapEnvelope& message : messagesChunk)
    {
        MessageData msg = convertPgEnvelope(message, imapFlags);
        destination.push_back(msg);
    }
}

Future<void> PgMessages::loadDetails(FolderRef mailbox, const seq_range& range)
{
    auto partialMessages = mailbox.filterByRanges(range, MessageData::isPartial);
    if (partialMessages->empty()) return makeFuture();

    auto mailboxInfo = mailbox.info();
    updateTargetRevision(mailboxInfo.revision);

    auto partialMids = std::make_shared<MidSet>(partialMessages->asMidSet());
    createMidToUid(partialMessages);

    return loadDetailsForMids(mailboxInfo, partialMids, ReplicaState::TryReplica)
        .then([=, capture_self](auto future) mutable {
            future.get();
            mailbox.insertMessages(details);
        });
}

Future<string> PgMessages::regenerateImapId(const std::string& mid)
{
    Promise<string> promise;
    imapRepo().imapRegenerateImapId(mid, [=, capture_self](auto&& err, auto&& res) mutable {
        if (err)
        {
            promise.set_exception(BackendError(err.full_message()));
            return;
        }
        promise.set(std::to_string(res.imapId));
    });
    return promise;
}

void PgMessages::createMidToUid(UidMapPtr partialMessages)
{
    midToUid.clear();
    while (!partialMessages->empty())
    {
        auto message = partialMessages->pop();
        midToUid.insert(midToUid.end(), std::pair<uint64_t, uint32_t>(message.mid, message.uid));
    }
}

Future<void> PgMessages::loadDetailsForMids(
    const FolderInfo& folderInfo,
    const std::shared_ptr<MidSet>& partialMids,
    ReplicaState replica)
{
    Promise<void> promise;
    ImapFolder folder = convertFolder(folderInfo);
    const size_t maxChunk =
        settings->maxDetailsLoadingChunk ? settings->maxDetailsLoadingChunk : 1000;
    auto chunker = std::make_shared<Chunker<MidSet, MidList>>(partialMids, maxChunk);
    auto operation = [=, capture_self](auto&& chunk) mutable -> Future<void> {
        Promise<void> promise;
        logLoadingMids(folder, chunk, replica);
        imapRepo(replica).imapGetDetails(
            folder, chunk, [=, capture_self](auto&& err, auto&& macsDetails) mutable {
                if (err)
                {
                    promise.set_exception(BackendError(err.full_message()));
                    return;
                }
                updateStatsOnLoadMessages(macsDetails.size());
                auto converted = convertPgDetails(macsDetails);
                details.insert(details.end(), converted.begin(), converted.end());
                promise.set();
            });
        return promise;
    };
    return applyToChunks(chunker, operation);
}

void PgMessages::logLoadingMids(const ImapFolder& folder, const MidList& mids, ReplicaState replica)
{
    ostringstream logStream;
    logStream << "Loading details chunk "
              << (replica == ReplicaState::UseReplica ? "from replica " : "") << "{" << folder.name
              << ", fid:" << folder.fid << ", rev:" << folder.revision << ", chunk:" << mids.size()
              << "}\r\n"
              << "[";
    size_t i = 0;
    for (auto mid : mids)
    {
        logStream << mid << " ";
        if (++i % 20 == 0) logStream << "\r\n";
    }
    logStream << "]";
    logger.log() << logStream.str();
}

MidSet PgMessages::getMissedMids(const MidSet& partialMids, MidSet&& loadedMids)
{
    MidSet result;
    if (loadedMids.size() < partialMids.size())
    {
        std::set_difference(
            partialMids.begin(),
            partialMids.end(),
            loadedMids.begin(),
            loadedMids.end(),
            std::inserter(result, result.end()));
    }
    return result;
}

uint32_t PgMessages::uidByMid(uint64_t mid) const
{
    auto found = midToUid.find(mid);
    return found != midToUid.end() ? found->second : 0;
}

MessageVector PgMessages::convertPgDetails(const ::macs::ImapEnvelopeDetailsChunk& detailsChunk)
{
    MessageVector details;
    details.reserve(detailsChunk.size());
    for (auto&& messageDetail : detailsChunk)
    {
        MessageData msg(uidByMid(messageDetail.envelope.mid), messageDetail.envelope.mid);
        msg.setDetails(
            messageDetail.mid,
            messageDetail.envelope.receivedDate,
            static_cast<uint32_t>(messageDetail.size));
        details.push_back(msg);
    }
    return details;
}

//------------------------------------------------------------------------------

Future<UidMapPtr> PgMessageId::loadByMessageId(
    const FolderInfo& folderInfo,
    const string& messageId)
{
    updateTargetRevision(folderInfo.revision);
    ImapFolder folder = convertFolder(folderInfo);

    logger.logDebug() << "Loading by message_id " << messageId << " from {" << folder.name
                      << ", fid:" << folder.fid << ", rev:" << folder.revision << "}";

    return loadImapFlags(ReplicaState::TryReplica).then([=, capture_self](auto future) mutable {
        imapFlags = future.get();
        resultMessages.reset(new UidMap());
        Promise<macs::ImapEnvelopeDetailsChunk> promise;
        imapRepo(ReplicaState::TryReplica)
            .imapGetByMessageId(
                folder, messageId, [=, capture_self](auto&& err, auto&& envelopes) mutable {
                    if (err)
                    {
                        promise.set_exception(BackendError(err.full_message()));
                    }
                    else
                    {
                        promise.set(envelopes);
                    }
                });
        return Future<ImapEnvelopeDetailsChunk>(promise).then([=, capture_self](
                                                                  auto&& future) mutable {
            auto envelopes = future.get();
            logger.logDebug() << "Response from DB by num: got " << envelopes.size() << " messages";
            updateStatsOnLoadMessages(envelopes.size());
            for (auto&& detailMsg : envelopes)
            {
                resultMessages->insert(convertPgEnvelope(detailMsg.envelope, imapFlags));
            }
            return resultMessages;
        });
    });
}

//------------------------------------------------------------------------------

Future<UidMapPtr> PgDeletedMessages::load(const FolderInfo& folderInfo)
{
    updateTargetRevision(folderInfo.revision);
    ImapFolder folder = convertFolder(folderInfo);
    resultMessages.reset(new UidMap());

    logger.logDebug() << "Loading deleted "
                      << " from {" << folder.name << ", fid:" << folder.fid
                      << ", rev:" << folder.revision << "}";

    Promise<macs::ImapEnvelopeChunk> promise;
    imapRepo(ReplicaState::TryReplica)
        .imapGetDeleted(folder, [=, capture_self](auto&& err, auto&& envelopes) mutable {
            if (err)
            {
                promise.set_exception(BackendError(err.full_message()));
            }
            else
            {
                promise.set(envelopes);
            }
        });
    return Future<macs::ImapEnvelopeChunk>(promise).then([=, capture_self](auto&& future) mutable {
        auto envelopes = future.get();
        logger.logDebug() << "Response from DB: got " << envelopes.size() << " deleted messages";
        updateStatsOnLoadMessages(envelopes.size());
        for (auto&& rawMsg : envelopes)
        {
            resultMessages->insert(convertPgEnvelope(rawMsg, imapFlags));
        }
        return resultMessages;
    });
}

//------------------------------------------------------------------------------

PgEnvelopes::PgEnvelopes(PgBackend& backend) : PgWorker(backend)
{
}

Future<void> PgEnvelopes::resetFreshCounter()
{
    Promise<void> promise;
    envelopesRepo().resetFreshCounter([promise](auto err, auto /*revision*/) mutable {
        if (err)
        {
            promise.set_exception(BackendError(err.full_message()));
            return;
        }
        promise.set();
    });
    return promise;
}

} // namespace backend
} // namespace yimap
