#pragma once

#include <common/settings.h>

#include <common/folder.h>

#include <backend/meta_common/helpers.h>
#include <backend/meta_pg/pg_backend.h>
#include <backend/meta_pg/pg_worker.h>

#include <macs/envelopes_repository.h>
#include <macs/labels_repository.h>
#include <macs/hooks.h>
#include <macs/imap_envelope.h>

#include <yplatform/find.h>

namespace yimap { namespace backend {

class PgMessages : public PgWorker
{
public:
    PgMessages(PgBackend& backend);

    Future<void> load(FolderRef mailbox, const seq_range& range);
    Future<void> loadChunk(FolderRef mailbox, const seq_range& range, size_t limit);
    Future<void> loadDetails(FolderRef mailbox, const seq_range& range);

    Future<string> regenerateImapId(const std::string& mid);

protected:
    template <typename MacsLoader>
    Future<void> loadMessages(
        const FolderInfo& folderInfo,
        const seq_range& rangeToCache,
        MacsLoader macsLoader)
    {
        using namespace ::macs;

        if (rangeToCache.empty()) return makeFuture();
        updateTargetRevision(folderInfo.revision);
        ImapFolder folder = convertFolder(folderInfo);

        uint32_t from = rangeToCache.first();
        uint32_t to = rangeToCache.last();

        logger.logDebug() << "Loading messages [" << from << ", " << to << "]"
                          << " from {" << folder.name << ", fid:" << folder.fid
                          << ", rev:" << folder.revision << "}";

        folder.revision = 0;
        Promise<void> promise;
        macsLoader(
            folder,
            from,
            to,
            ImapRepository::ImapListMode_byUid,
            [=, capture_self](auto&& err, auto&& envelopes) mutable {
                if (err)
                {
                    promise.set_exception(BackendError(err.full_message()));
                    return;
                }
                logger.logDebug() << "Response from DB: got " << envelopes.size() << " messages";
                updateStatsOnLoadMessages(envelopes.size());
                convertMessages(envelopes, messages);
                promise.set();
            });
        return promise;
    }
    void logLoadingMids(
        const ::macs::ImapFolder& folder,
        const MidList& mids,
        ReplicaState replica);
    Future<void> loadDetailsForMids(
        const FolderInfo& folderInfo,
        const std::shared_ptr<MidSet>& partialMids,
        ReplicaState replica);
    void convertMessages(
        const ::macs::ImapEnvelopeChunk& messagesChunk,
        MessageVector& destination);

    void createMidToUid(UidMapPtr partialMessages);

    MessageVector convertPgDetails(const ::macs::ImapEnvelopeDetailsChunk& detailsChunk);
    uint32_t uidByMid(uint64_t mid) const;

    static MidSet getMissedMids(const MidSet& partialMids, MidSet&& loadedMids);

    seq_range takeChunk(const seq_range& range)
    {
        auto res = seq_range(range.minVal(), range.maxVal(), range.uidMode());
        res += *range.begin();
        return res;
    }

    void updateStatsOnLoadMessages(int64_t count)
    {
        context->messagesStats.loadedFromMdbCount += count;
    }

protected:
    ::macs::LabelSet imapFlags;
    ImapContextPtr context;

    FolderInfo mailboxInfo;
    MessageVector messages;
    MessageVector details;
    std::map<uint64_t, uint32_t> midToUid;
};

class PgMessageId : public PgMessages
{
public:
    PgMessageId(PgBackend& backend) : PgMessages(backend)
    {
    }
    Future<UidMapPtr> loadByMessageId(const FolderInfo& folderInfo, const string& messageId);

private:
    UidMapPtr resultMessages;
};

class PgDeletedMessages : public PgMessages
{
public:
    PgDeletedMessages(PgBackend& backend) : PgMessages(backend)
    {
    }
    Future<UidMapPtr> load(const FolderInfo& folderInfo);

private:
    UidMapPtr resultMessages;
};

class PgEnvelopes : public PgWorker
{
public:
    PgEnvelopes(PgBackend& backend);

    Future<void> resetFreshCounter();
};

} // namespace backend
} // namespace yimap
