#pragma once

#include <backend/meta_pg/pg_worker.h>
#include <common/user_settings.h>
#include <common/types.h>
#include <yplatform/util/string_translation.h>
#include <yplatform/util.h>
#include <boost/date_time/gregorian/gregorian.hpp>

namespace yimap { namespace backend {

class PgSettings : public PgWorker
{
public:
    inline static string imapEnabledSetting = "enable_imap"s;
    inline static string renameEnabledSetting = "imap_rename_enabled"s;
    inline static string localizeSetting = "localize_imap"s;
    inline static string autoexpungeDisabledSetting = "disable_imap_autoexpunge"s;
    inline static string clientLogEnabledDateSetting = "client_log_enable_date"s;
    inline static string enableAuthPlainSetting = "enable_imap_auth_plain"s;

    PgSettings(PgBackend& backend) : PgWorker(backend)
    {
    }

    Future<UserSettings> load()
    {
        Promise<UserSettings> promise;
        settingsRepo(ReplicaState::UseReplica)
            .getSettings(
                requiredSettings(),
                [=, capture_self](auto&& err, macs::SettingsPtr settings) mutable {
                    if (err)
                    {
                        return promise.set_exception(BackendError(err.full_message()));
                    }
                    promise.set(makeUserSettings(settings));
                });
        return promise;
    }

    const std::vector<string>& requiredSettings()
    {
        static std::vector<string> list = {
            imapEnabledSetting,          renameEnabledSetting,       localizeSetting,
            clientLogEnabledDateSetting, autoexpungeDisabledSetting, enableAuthPlainSetting
        };
        return list;
    }

    UserSettings makeUserSettings(macs::SettingsPtr settings)
    {
        const macs::SettingsMap& profile = settings->profile.single_settings;
        const macs::SettingsMap& parameters = settings->parameters.single_settings;
        UserSettings ret;
        ret.imapEnabled = extractBoolSetting(imapEnabledSetting, ret.imapEnabled, profile);
        ret.renameEnabled = extractBoolSetting(renameEnabledSetting, ret.renameEnabled, profile);
        ret.localizeImap = extractBoolSetting(localizeSetting, ret.localizeImap, parameters);
        ret.enableAutoExpunge =
            !extractBoolSetting(autoexpungeDisabledSetting, !ret.enableAutoExpunge, parameters);
        ret.enableAuthPlain =
            extractBoolSetting(enableAuthPlainSetting, ret.enableAuthPlain, parameters);
        auto clientLogEnableDate = extractDateSetting(clientLogEnabledDateSetting, parameters);
        if (clientLogEnableDate)
        {
            boost::gregorian::date currentDate(boost::gregorian::day_clock::local_day());
            ret.clientLogEnabled = currentDate <= *clientLogEnableDate;
        }
        return ret;
    }

    bool extractBoolSetting(const string& key, bool defaultValue, const macs::SettingsMap& settings)
    {
        auto it = settings.find(key);
        if (it == settings.end()) return defaultValue;
        return it->second == "on" || it->second == "true" ? true : false;
    }

    std::optional<boost::gregorian::date> extractDateSetting(
        const string& key,
        const macs::SettingsMap& settings)
    {
        auto it = settings.find(key);
        if (it == settings.end() || it->second == "") return {};
        try
        {
            return boost::gregorian::from_string(it->second);
        }
        catch (const std::exception& e)
        {
            logger.logWarning() << "parse \"" << key << "\" error: " << e.what();
            return {};
        }
    }
};

}}
