#pragma once

#include <common/log/logger.h>
#include <common/imap_context.h>
#include <common/settings.h>
#include <backend/backend_types.h>
#include <macs_pg/macs_pg.h>

namespace yimap { namespace backend {

class PgBackend;

class PgWorker : public std::enable_shared_from_this<PgWorker>
{
public:
    PgWorker(PgBackend& backend);
    virtual ~PgWorker()
    {
    }

    enum class ReplicaState
    {
        UseMaster,
        TryReplica,
        UseReplica
    };

protected:
    Future<::macs::LabelSet> loadImapFlags(ReplicaState replica = ReplicaState::UseMaster)
    {
        Promise<::macs::LabelSet> promise;
        labelsRepo(replica).resetLabelsCache();
        labelsRepo(replica).getAllLabels([capture_self, promise](auto err, auto allLabels) mutable {
            if (err) return promise.set_exception(BackendError(err.message()));
            ::macs::LabelSet imapFlags;
            for (auto&& label : allLabels)
            {
                if (label.second.type() == macs::Label::Type::system ||
                    label.second.type() == macs::Label::Type::imap)
                {
                    imapFlags.insert(label);
                }
            }
            promise.set(imapFlags);
        });
        return promise;
    }

    macs::ImapRepository& imapRepo(ReplicaState replica = ReplicaState::UseMaster)
    {
        // TODO: check if we can use replica
        return pgService(replica).imapRepo();
    }

    macs::LabelsRepository& labelsRepo(ReplicaState replica = ReplicaState::UseMaster)
    {
        return pgService(replica).labels();
    }

    macs::FoldersRepository& foldersRepo(ReplicaState replica = ReplicaState::UseMaster)
    {
        return pgService(replica).folders();
    }

    macs::EnvelopesRepository& envelopesRepo(ReplicaState replica = ReplicaState::UseMaster)
    {
        return pgService(replica).envelopes();
    }

    macs::SettingsRepository& settingsRepo(ReplicaState replica = ReplicaState::UseMaster)
    {
        return pgService(replica).settings();
    }

    ::macs::Service& pgService(ReplicaState replica = ReplicaState::UseMaster)
    {
        if (replica == ReplicaState::UseMaster) return pgMaster();

        if (replica == ReplicaState::UseReplica) return pgReplica();

        // Last case: ReplicaState::TryReplica
        // We should check if replica revision is big enough
        if (replicaRevision() >= masterTargetRevision())
        {
            return pgReplica();
        }
        return pgMaster();
    }

    ::macs::Service& pgMaster();

    ::macs::Service& pgReplica();

    ::macs::Revision replicaRevision();

    ::macs::Revision masterTargetRevision() const
    {
        if (knownTargetRevision == ::macs::NULL_REVISION)
        {
            throw std::runtime_error("knownTargetRevision should not be NULL_REVISION.");
        }
        return knownTargetRevision;
    }

    void updateTargetRevision(::macs::Revision newRevision)
    {
        knownTargetRevision = std::max(knownTargetRevision, newRevision);
    }

protected:
    PgBackend& backend;
    ImapContextPtr context;
    Logger& logger;
    SettingsCPtr settings;
    ::macs::ServicePtr pg;
    ::macs::ServicePtr pgReplicaService;

    ::macs::Revision knownTargetRevision = ::macs::NULL_REVISION;
    ::macs::Revision knownReplicaRevision = ::macs::NULL_REVISION;
};

//-----------------------------------------------------------------------------
// PG helpers

MessageData convertPgEnvelope(
    const ::macs::ImapEnvelope& message,
    const ::macs::LabelSet& imapFlags);

MessagesVectorPtr convertPgChanges(
    const ::macs::ImapEnvelopeChangesChunk& changes,
    const ::macs::LabelSet& imapFlags);

template <typename Container, typename ChunkContainer = Container>
class Chunker
{
    std::shared_ptr<Container> source;
    typename Container::const_iterator from;
    const typename Container::const_iterator to;
    const size_t chunkSize;

public:
    Chunker(const std::shared_ptr<Container>& source, size_t chunkSize)
        : source(source), from(source->begin()), to(source->end()), chunkSize(chunkSize)
    {
    }

    bool empty() const
    {
        return from == to;
    }

    template <typename DestCountainer = ChunkContainer>
    DestCountainer next()
    {
        DestCountainer result;
        for (size_t i = 0; i < chunkSize && from != to; from++, i++)
        {
            result.push_back(*from);
        }
        return result;
    }
};

template <typename Chunker, typename Operation>
inline Promise<void> applyToChunks(Chunker chunker, Operation operation)
{
    Promise<void> promise;
    auto applyOp = [chunker, operation, promise](auto continuation) mutable {
        if (chunker->empty()) return promise.set();
        auto chunk = chunker->next();
        auto future = operation(chunk);
        future.add_callback([future, continuation, promise]() mutable {
            if (future.has_exception())
            {
                return promise.set_exception(future.get_exception());
            }
            continuation(continuation);
        });
    };
    applyOp(applyOp);
    return promise;
}

template <typename Iterator, typename Operation>
inline Promise<void> applyToRange(Iterator begin, Iterator end, Operation operation)
{
    Promise<void> promise;
    auto applyOp = [operation, promise](Iterator begin, Iterator end, auto continuation) mutable {
        if (begin == end) return promise.set();
        auto future = operation(*begin);
        ++begin;
        future.add_callback([future, continuation, promise, begin, end]() mutable {
            if (future.has_exception())
            {
                return promise.set_exception(future.get_exception());
            }
            continuation(begin, end, continuation);
        });
    };
    applyOp(begin, end, applyOp);
    return promise;
}

} // namespace backend
} // namespace yimap
