#include "spam_report.h"

#include <common/imap_context.h>
#include <yplatform/log.h>
#include <yplatform/log/typed.h>
#include <common/flags.h>
#include <common/message_set.h>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/join.hpp>

namespace yimap {

const std::string SpamReport::ACTION_FLAG = "flag";
const std::string SpamReport::ACTION_COPY = "copy";
const std::string SpamReport::ACTION_MOVE = "move";

SpamReport::SpamReport(
    const ImapContext& context,
    const UidMap& uids,
    const BodyMetadataByMid& bodyMeta,
    const std::string type,
    const std::string& fromFolder,
    const std::string toFolder,
    const std::string& flag)
    : sessionId(context.uniq_id())
    , uid(context.userData.uid)
    , suid(context.userData.suid)
    , ip(context.sessionInfo.remoteAddress)
    , fromFolder(fromFolder)
    , toFolder(toFolder)
    , type(type)
    , flag(flag)
    , client(context.userData.clientId.raw)
{
    uids.iterate([this, &bodyMeta](const MessageData& message, uint32_t) {
        midList.push_back({ message.smid(),
                            bodyMeta.at(message.smid()).stid,
                            message.time,
                            message.has_flag(Flags::MSG_SEEN) });
    });
}

std::unique_ptr<SpamReport> SpamReport::make(
    const ImapContext& context,
    const UidMap& uids,
    const BodyMetadataByMid& bodyMeta,
    const std::string type,
    const std::string& fromFolder,
    const std::string toFolder,
    const std::string& flag)
{
    return std::unique_ptr<SpamReport>(
        new SpamReport(context, uids, bodyMeta, type, fromFolder, toFolder, flag));
}

void SpamReport::logToTskv() const
{
    auto tskvLogger = yplatform::log::tskv_logger(YGLOBAL_LOG_SERVICE, "so_report");

    namespace ll = yplatform::log::typed;

    for (auto seenMid = midList.begin(); seenMid != midList.end(); seenMid++)
    {
        YLOG(tskvLogger, info)
            << ll::make_attr("session", sessionId) << ll::make_attr("uid", uid)
            << ll::make_attr("suid", suid) << ll::make_attr("ip", ip)
            << ll::make_attr("source", "imap")
            << ll::make_attr("type", (type == ACTION_FLAG) ? (type + ":" + flag) : type)
            << ll::make_attr("folder", fromFolder) << ll::make_attr("dest_folder", toFolder)
            << ll::make_attr("client", boost::replace_all_copy(client, "\t", " "))
            << ll::make_attr("mid", seenMid->mid)
            << ll::make_attr("seen", seenMid->seen ? "yes" : "no")
            << ll::make_attr("stid", seenMid->stid)
            << ll::make_attr("received_date", seenMid->receiveTime);
    }
}

} // namespace yimap
