#include "msearch.h"

#include <common/uid_map.h>
#include <common/types.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/property_tree/json_parser.hpp>

namespace yimap {

struct SearchOp
{
    using YieldContext = yplatform::yield_context<SearchOp>;
    using Handler = SearchBackend::SearchCallback;

    ImapContextPtr ctx;
    string searchRequest;
    string originalRequest;
    string fid;
    Handler handler;

    std::set<string> mids = {};
    size_t respondedMidsSize = 0;
    size_t offset = 0;
    size_t chunkSize = 5000;

    void operator()(
        YieldContext yieldСtx,
        ErrorCode ec = {},
        yhttp::response response = yhttp::response())
    {
        try
        {
            reenter(yieldСtx)
            {
                do
                {
                    offset = mids.size();
                    yield search(yieldСtx);
                    if (ec)
                    {
                        LERR_(ctx) << "search op: message=" << ec.message();
                        yield break;
                    }
                    if (response.status != 200)
                    {
                        LERR_(ctx) << "search op: status= " << response.status
                                   << " reason=" << response.reason;
                        ec = ymod_httpclient::http_error::code::server_status_error;
                        yield break;
                    }
                    respondedMidsSize = parseResponse(response.body);
                } while (respondedMidsSize >= chunkSize);
            }
        }
        catch (const std::exception& e)
        {
            LERR_(ctx) << "search op exception: " << e.what();
            ec = boost::asio::error::operation_aborted;
        }

        if (yieldСtx.is_complete())
        {
            handler(ec, mids);
        }
    }

    void search(YieldContext yieldСtx)
    {
        auto request = yhttp::request::GET(
            "/api/async/mail/search" +
            yhttp::url_encode({ { "get", "mid" },
                                { "imap", "1" },
                                { "remote_ip", ctx->sessionInfo.remoteAddress },
                                { "uid", ctx->userData.uid },
                                { "fid", fid },
                                { "user_request", originalRequest },
                                { "request", searchRequest },
                                { "first", offset },
                                { "count", chunkSize } }));
        auto client = yplatform::find<yhttp::cluster_client>("search_client");
        client->async_run(ctx, std::move(request), yieldСtx);
    }

    size_t parseResponse(const std::string& response)
    {
        std::stringbuf buf(response);
        std::istream stream(&buf);
        Ptree ptree;
        boost::property_tree::read_json(stream, ptree);
        size_t respondedMidsSize = 0;
        auto envelopes = ptree.get_child("envelopes");
        for (auto& [name, mid] : envelopes)
        {
            ++respondedMidsSize;
            mids.insert(mid.data());
        }
        return respondedMidsSize;
    }
};

void HTTPMSearchBackend::search(
    const string& searchRequest,
    const string& originalRequest,
    const string& fid,
    const SearchCallback& cb)
{
    yplatform::spawn(
        context->ioService, SearchOp{ context, searchRequest, originalRequest, fid, cb });
}

}

#include <yplatform/unyield.h>
