#pragma once

#include <backend/backend.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>

namespace yimap::backend {

class HTTPUserSettingsBackend : public UserSettingsBackend
{
public:
    HTTPUserSettingsBackend(ImapContextPtr context) : context(context)
    {
    }

    Future<UserSettings> loadSettings() override
    {
        Promise<UserSettings> promise;
        auto client = yplatform::find<yhttp::cluster_client>(context->ioService, "settings_client");
        client->async_run(
            context,
            prepareHTTPRequest(),
            [context = context, promise](auto ec, auto response) mutable {
                try
                {
                    if (ec)
                    {
                        return promise.set_exception(
                            LoadSettingsError(context->userData.uid, ec.message()));
                    }

                    if (response.status != 200)
                    {
                        return promise.set_exception(LoadSettingsError(
                            context->userData.uid, "status:"s + std::to_string(response.status)));
                    }

                    promise.set(UserSettingsFromJsonStr(response.body));
                }
                catch (const std::exception& e)
                {
                    promise.set_exception(e);
                }
            });
        return promise;
    }

    yhttp::request prepareHTTPRequest()
    {
        static const std::string settingsList = "enable_imap\r"
                                                "imap_rename_enabled\r"
                                                "localize_imap\r"
                                                "client_log_enable_date\r"
                                                "disable_imap_autoexpunge\r"
                                                "enable_imap_auth_plain";
        auto request = yhttp::request::GET(
            "/get" +
            yhttp::url_encode({ { "uid", context->userData.uid },
                                { "suid", context->userData.suid },
                                { "mdb", context->userData.storage },
                                { "settings_list", settingsList },
                                { "db_role", "replica" },
                                { "format", "json" } }));
        return request;
    }

    static UserSettings UserSettingsFromJsonStr(string& src)
    {
        std::stringbuf buf(src);
        std::istream stream(&buf);
        Ptree ptree;
        boost::property_tree::read_json(stream, ptree);
        return UserSettingsFromPtree(ptree);
    }

    static UserSettings UserSettingsFromPtree(const Ptree& ptree)
    {
        UserSettings settings;
        auto profileSettings = ptree.get_child("settings.profile.single_settings");
        settings.imapEnabled = profileSettings.get("enable_imap", settings.imapEnabled);
        settings.renameEnabled = profileSettings.get("imap_rename_enabled", settings.renameEnabled);
        auto parametersSettings = ptree.get_child("settings.parameters.single_settings");
        settings.localizeImap = parametersSettings.get("localize_imap", settings.localizeImap);
        settings.enableAutoExpunge =
            !parametersSettings.get("disable_imap_autoexpunge", !settings.enableAutoExpunge);
        settings.enableAuthPlain =
            parametersSettings.get("enable_imap_auth_plain", settings.enableAuthPlain);
        auto clientLogDate = parametersSettings.get("client_log_enable_date", "");
        if (clientLogDate.size())
        {
            boost::gregorian::date logDate = boost::gregorian::from_string(clientLogDate);
            boost::gregorian::date currentDate(boost::gregorian::day_clock::local_day());
            settings.clientLogEnabled = currentDate <= logDate;
        }
        return settings;
    }

private:
    ImapContextPtr context;
};
}