#pragma once

#include <common/settings.h>
#include <common/types.h>

namespace yimap { namespace backend {

class ThrottlingController
{
public:
    ThrottlingController(ThrottlingSettings settings) : settings(settings)
    {
    }

    Duration recoveryDelay()
    {
        if (consumedCount >= settings.limit)
        {
            return consumedResetTime() - Clock::now();
        }

        return Milliseconds(0);
    }

    uint64_t limit()
    {
        if (Clock::now() >= consumedResetTime())
        {
            resetConsumed();
        }

        if (settings.limit <= consumedCount) return 0;
        return settings.limit - consumedCount;
    }

    void consume(size_t count)
    {
        consumedCount += count;
    }

private:
    void resetConsumed()
    {
        consumedCountResetTs = Clock::now();
        consumedCount = 0;
    }

    TimePoint consumedResetTime()
    {
        auto delayMultiplier = consumedCount / static_cast<double>(settings.limit);
        if (delayMultiplier < 1)
        {
            delayMultiplier = 1;
        }

        auto delay = settings.windowLength * delayMultiplier;
        return consumedCountResetTs + std::min(durationCast<Duration>(delay), settings.maxDelay);
    }

    ThrottlingSettings settings;
    size_t consumedCount = 0;
    TimePoint consumedCountResetTs = Clock::now();
};

using ThrottlingControllerPtr = std::shared_ptr<ThrottlingController>;

}}
