#include "backend.h"
#include "dispatcher.h"
#include <boost/property_tree/json_parser.hpp>

namespace yimap::backend {

XivaNotificationsBackend::XivaNotificationsBackend(
    ImapContextPtr context,
    XivaDispatcherPtr dispatcher,
    ControlConfigPtr config)
    : context(context)
    , ioService(context->ioService)
    , dispatcher(dispatcher)
    , config(config)
    , xivaLog(YGLOBAL_LOG_SERVICE, "xiva_log")
{
}

XivaNotificationsBackend::~XivaNotificationsBackend()
{
    if (controller)
    {
        ioService.post([dispatcher = dispatcher,
                        controller = controller,
                        uid = userId(),
                        session = sessionId()]() {
            dispatcher->unsubscribe(uid, session);
            controller->stop();
        });
    }
}

void XivaNotificationsBackend::subscribe(WeakEventHandlerPtr handler)
{
    ioService.post([this, capture_self, handler]() {
        spawnControllerOnce();
        addWeakHandler(handler);
    });
}

Future<bool> XivaNotificationsBackend::handleEvent(const XivaEvent& xivaEvent)
{
    Promise<bool> resPromise;
    ioService.post([this, capture_self, xivaEvent, resPromise]() mutable {
        ImapEventPtr event = createImapEvent(xivaEvent);
        if (controller && xivaEvent.subscriptionId == controller->subscriptionId())
        {
            resPromise.set(processEvent(event));
        }
        else
        {
            resPromise.set(false);
        }
    });
    return resPromise;
}

void XivaNotificationsBackend::addHandler(EventHandlerPtr handler)
{
    ioService.post([this, capture_self, handler]() { handlers.insert(handler); });
}

void XivaNotificationsBackend::addWeakHandler(WeakEventHandlerPtr handler)
{
    addHandler(std::make_shared<WeakHandlerProxy>(handler));
}

const string& XivaNotificationsBackend::userId() const
{
    return context->userData.uid;
}

const string& XivaNotificationsBackend::sessionId() const
{
    return context->uniq_id();
}

ImapEventPtr XivaNotificationsBackend::createImapEvent(const XivaEvent& xivaEvent)
{
    if (config->events.count(xivaEvent.operation))
    {
        try
        {
            Ptree data;
            stringstream stream(xivaEvent.raw_data);
            boost::property_tree::read_json(stream, data);

            auto connectionId = data.get("connection_id", "");
            auto action = config->events.at(xivaEvent.operation);
            ImapEventPtr event(
                new ImapTimedEvent(xivaEvent.operation, action, xivaEvent.eventTs, connectionId));
            return event;
        }
        catch (const std::exception& e)
        {
            YLOG(xivaLog, error) << "exception in createImapEvent: " << e.what();
        }
    }

    ImapEventPtr event(new ImapUnconditionalEvent(xivaEvent.operation));
    return event;
}

bool XivaNotificationsBackend::processEvent(ImapEventPtr event)
{
    std::set<EventHandlerPtr> victims;
    for (auto&& handler : handlers)
    {
        auto keepHandler = handler->onEvent(event);
        if (!keepHandler) victims.insert(handler);
    }
    for (auto&& handlerToDelete : victims)
    {
        handlers.erase(handlerToDelete);
    }
    return handlers.size() > 0;
}

void XivaNotificationsBackend::spawnControllerOnce()
{
    if (!controller)
    {
        controller = std::make_shared<SubscriptionControl>(context, config);
        controller->start();
        dispatcher->subscribe(userId(), sessionId(), shared_from(this));
    }
}

}
