#include "dispatcher.h"
#include "xiva_event.h"
#include "backend.h"
#include <yplatform/future/multi_future.hpp>

namespace yimap::backend {

void XivaDispatcher::subscribe(
    const string& uid,
    const string& /* sessionId */,
    XivaNotificationsBackendWeakPtr handler)
{
    std_lock lock(mux);
    handlers.insert(HandlersMap::value_type(uid, handler));
}

void XivaDispatcher::unsubscribe(const string& uid, const string& sessionId)
{
    std_lock lock(mux);

    list<HandlersMap::iterator> victims;
    pair<HandlersMap::iterator, HandlersMap::iterator> range = handlers.equal_range(uid);
    for (HandlersMap::iterator it = range.first; it != range.second; it++)
    {
        XivaNotificationsBackendPtr realHandler(it->second.lock());
        if (!realHandler || sessionId == realHandler->sessionId()) victims.push_back(it);
    }
    for (auto victim : victims)
    {
        handlers.erase(victim);
    }
}

template <typename T>
void unpack(const string& source, T& message)
{
    msgpack::unpacked unpacked_msg;
    msgpack::unpack(unpacked_msg, source.data(), source.size());
    unpacked_msg.get().convert(message);
}

void XivaDispatcher::process(ymod_webserver::http::stream_ptr stream)
{
    XivaEvent event;

    try
    {
        auto req = stream->request();
        const string requestBody = string(req->raw_body.begin(), req->raw_body.end());
        unpack(requestBody, event);

        auto found = req->url.params.find("subscription_id");
        if (found != req->url.params.end())
        {
            event.subscriptionId = found->second;
        }

        handleEvent(std::move(event), stream);
    }
    catch (const std::exception& e)
    {
        YLOG(xivaLog, error) << "Got http xiva notify, but exception: " << e.what();
    }
    catch (...)
    {
    }
}

void XivaDispatcher::respondProcessed(
    ymod_webserver::http::stream_ptr stream,
    bool processed,
    const string& subscriptionId)
{
    if (processed)
    {
        stream->result(ymod_webserver::codes::ok, "OK\r\n");
    }
    else
    {
        stream->result(ymod_webserver::codes::reset_content, "\r\n");
        YLOG(xivaLog, error) << "Xiva event " << subscriptionId
                             << " not handled. Sending 205 response.";
    }
}

// TODO respond immediately?
void XivaDispatcher::handleEvent(XivaEvent&& xivaEvent, ymod_webserver::http::stream_ptr stream)
{
    list<XivaNotificationsBackendPtr> realHandlers;
    list<HandlersMap::iterator> deadHandlers;
    {
        std_lock lock(mux);
        pair<HandlersMap::iterator, HandlersMap::iterator> range =
            handlers.equal_range(xivaEvent.uid);
        for (HandlersMap::iterator it = range.first; it != range.second; it++)
        {
            XivaNotificationsBackendPtr realHandler(it->second.lock());
            if (realHandler) realHandlers.push_back(realHandler);
            else
                deadHandlers.push_back(it);
        }
        for (auto victim : deadHandlers)
        {
            handlers.erase(victim);
        }
    }

    std::vector<Future<bool>> futures;
    for (auto handler : realHandlers)
    {
        futures.push_back(handler->handleEvent(xivaEvent));
    }

    auto combination = yplatform::future::future_multi_and(futures);
    combination.add_callback([stream,
                              futures = std::move(futures),
                              xivaEvent = std::move(xivaEvent),
                              combination,
                              this,
                              self = shared_from_this()]() {
        bool eventProcessed = false;
        for (auto& future : futures)
        {
            try
            {
                eventProcessed = future.get() || eventProcessed;
            }
            catch (const std::exception& e)
            {
                YLOG(xivaLog, error) << "Got exception during handleEvent: " << e.what();
            }
        }
        respondProcessed(stream, eventProcessed, xivaEvent.subscriptionId);
    });
}

}