#pragma once

#include "subscription.h"
#include <backend/yreflection_types.h>
#include <common/imap_context.h>
#include <common/types.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yimap::backend {

struct SubscriptionControl : std::enable_shared_from_this<SubscriptionControl>
{
    using YieldCtx = yplatform::yield_context<SubscriptionControl>;

    ImapContextWeakPtr weakContext;
    string userId;
    string sessionId;
    ControlConfigPtr config;
    Timer timer;
    ErrorCode err;
    Subscription subscription;
    yplatform::log::source xivaLog;

    SubscriptionControl(ImapContextPtr context, ControlConfigPtr config)
        : weakContext(context)
        , userId(context->userData.uid)
        , sessionId(context->uniq_id())
        , config(config)
        , timer(context->ioService)
        , xivaLog(YGLOBAL_LOG_SERVICE, "xiva_log")
    {
    }

    void start()
    {
        if (auto context = weakContext.lock())
        {
            yplatform::spawn(context->ioService, shared_from(this));
        }
        else
        {
            throw std::logic_error("cannot start controller without context");
        }
    }

    void stop()
    {
        timer.get_io_service().post([this, capture_self]() {
            timer.cancel();
            weakContext.reset();
        });
    }

    void operator()(YieldCtx yieldCtx)
    {
        auto context = weakContext.lock();

        reenter(yieldCtx)
        {
            while (true)
            {
                yield subscribe(yieldCtx.capture(err, subscription));
                if (!context) break;
                yield wait(yieldCtx.capture(err));
                if (err == boost::asio::error::operation_aborted) break;
                if (!context) break;
            }
            yield unsubscribe(yieldCtx.capture(err));
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            reportException(e.what());
        }
    }

    template <typename Handler>
    void subscribe(Handler handler)
    {
        auto request = buildRequest(
            "/v2/subscribe/url",
            { { "service", config->service },
              { "user", userId },
              { "client", "imap" },
              { "session", sessionId },
              { "callback", config->callback } });

        findHTTPClient()->async_run(
            Context::fake(), request, [this, handler](auto err, auto response) mutable {
                static const ErrorCode tryAgainError = boost::asio::error::try_again;
                if (err)
                {
                    reportSubscribeFailed(err);
                    return handler(err, Subscription{});
                }
                if (response.status != 200)
                {
                    reportSubscribeFailed(err, response.status, response.body);
                    return handler(tryAgainError, Subscription{});
                }

                Subscription result;
                try
                {
                    result = parseSubscription(response.body);
                }
                catch (...)
                {
                    handler(tryAgainError, Subscription{});
                }
                handler(ErrorCode{}, result);
            });
    }

    template <typename Handler>
    void unsubscribe(Handler handler)
    {
        if (!subscribed()) return handler(ErrorCode{});

        auto request = buildRequest(
            "/v2/unsubscribe/url",
            { { "service", config->service },
              { "user", userId },
              { "subscription_id", subscription.id } });

        findHTTPClient()->async_run(
            Context::fake(), request, [this, handler](auto err, auto response) mutable {
                if (err || response.status != 200)
                {
                    reportUnsubscribeFailed(err, response.status, response.body);
                }
            });
    }

    template <typename Handler>
    void wait(Handler handler)
    {
        TimePoint expires;

        // TODO move constants to config
        if (subscribed())
        {
            expires = Clock::now() + Hours(subscription.ttl) - Minutes(5);
        }
        else
        {
            expires = Clock::now() + Minutes(2);
        }

        timer.expires_at(expires);
        timer.async_wait(handler);
    }

    bool subscribed() const
    {
        return subscription.id.size();
    }

    const string& subscriptionId() const
    {
        // TODO check IO threads
        return subscription.id;
    }

    yhttp::request buildRequest(
        string&& url,
        ymod_httpclient::params_initializer_list&& queryParams)
    {
        std::ostringstream authHeader;
        authHeader << "Authorization: Xiva " << config->ctoken << "\r\n";
        return yhttp::request::POST(
            url + ymod_httpclient::url_encode(std::move(queryParams)), authHeader.str(), string{});
    }

    std::shared_ptr<yhttp::cluster_client> findHTTPClient()
    {
        return yplatform::find<yhttp::cluster_client, std::shared_ptr>("xiva_client");
    }

    Subscription parseSubscription(const string& data)
    {
        return yamail::data::deserialization::fromJson<Subscription>(data);
    }

    void reportSubscribeFailed(ErrorCode err, int status = 0, const string& body = {})
    {
        YLOG(xivaLog, error) << "subscribe failed: err=" << err.message() << " status=" << status
                             << " body=" << body;
    }

    void reportUnsubscribeFailed(ErrorCode err, int status = 0, const string& body = {})
    {
        YLOG(xivaLog, error) << "unsubscribe failed: err=" << err.message() << " status=" << status
                             << " body=" << body;
    }

    void reportException(const string& what)
    {
        YLOG(xivaLog, error) << "subscription control exception=" << what;
    }
};
}

#include <yplatform/unyield.h>
