#include "imap_command.h"
#include <backend/append/append.h>
#include <common/errors.h>
#include <common/quoted_string.h>

#include <yplatform/util/date_parse.h>
#include <yplatform/yield.h>
#include <functional>

namespace yimap {

struct Append : ImapAuthenticatedCommand
{
    using YieldCtx = yplatform::yield_context<Append>;

    Append(ImapCommandArgs& args) : ImapAuthenticatedCommand(args)
    {
    }

    ~Append()
    {
        updateStatsOnReleaseBuffer(messageBody.size());
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            extractArgsFromAST();

            if (!checkEncoding(destFolderName))
            {
                completeBad("[CLIENTBUG]", "Folder encoding error.");
                yield break;
            }

            folderInfoFuture = loadFolderInfo(destFolderName);
            yield folderInfoFuture.add_callback(yieldCtx);
            destFolder = folderInfoFuture.get();
            if (sharedFolder(destFolder))
            {
                completeNo("[UNAVAILABLE]", "Cannot APPEND to shared folder.");
                yield break;
            }

            yield append(makeAppendRequest()).then(yieldCtx.capture(appendResult));
            extractAppendResponse();

            if (messageIsDuplicate())
            {
                yield generateNewUid(appendedMid).then(yieldCtx.capture(appendedUid));
            }

            yield updateFolderAndSendDiff().then(yieldCtx);
            complete();
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const NoSuchFolderError&)
        {
            completeNo("[TRYCREATE]", "No such folder.");
        }
        catch (const ParseDateError&)
        {
            // serverbug - because this kind of error should be intercepted in
            // command parser
            logError() << "append exception: parse date error";
            completeBad("[SERVERBUG]", "Invalid arguments, cannot parse date-time.");
        }
        catch (const NoBodyError&)
        {
            logError() << "append exception: body not found";
            completeBad("[SERVERBUG]", "Invalid arguments, cannot find message part.");
        }
        catch (const std::exception& e)
        {
            logError() << "append exception: " << e.what();
            completeNo("[UNAVAILABLE]", "Backend error.");
        }
    }

    void extractArgsFromAST()
    {
        Trees& cmdArgs = tree->data.trees.front().children;
        destFolderName = quotedArg(1);
        messageDateTime = getDate(cmdArgs);
        messageBody = getBody(cmdArgs);
        updateStatsOnFillBuffer(messageBody.size());
        getFlags(cmdArgs, systemFlags, userFlags);
    }

    bool sharedFolder(const FolderInfo& folder)
    {
        return imapContext->foldersCache.getSharedStatusByFid(folder.fid);
    }

    string getDate(Trees& cmdArgs)
    {
        const TreeNode* dateTimeNode = 0;
        for (size_t idx = 1; idx < cmdArgs.size(); ++idx)
        {
            if (cmdArgs[idx].value.id().to_long() == lex_ids::DATE_TIME)
            {
                dateTimeNode = &cmdArgs[idx];
                break;
            }
        }

        ::time_t internalTime = ::time(0);
        if (dateTimeNode)
        {
            yplatform::util::date_full date;
            if (!yplatform::util::parse_date_full(
                    date, dateTimeNode->value.begin(), dateTimeNode->value.end()))
            {
                throw ParseDateError();
            }

            struct ::tm tm;
            std::memset(&tm, 0, sizeof tm);
            tm.tm_sec = date.sec;
            tm.tm_min = date.min;
            tm.tm_hour = date.hour;
            tm.tm_mday = date.dd;
            tm.tm_mon = date.mm;
            tm.tm_year = date.yyyy - 1900;

            internalTime = timegm(&tm);

            internalTime -= (3600 * (date.zone / 100) + 60 * (date.zone % 100));
        }
        return boost::lexical_cast<string>(internalTime);
    }

    string getBody(Trees& cmdArgs)
    {
        const TreeNode* messageNode = 0;
        for (size_t idx = 1; idx < cmdArgs.size(); ++idx)
        {
            if (cmdArgs[idx].value.id().to_long() == lex_ids::RFC822_MESSAGE)
            {
                messageNode = &cmdArgs[idx];
                break;
            }
        }

        if (!messageNode)
        {
            throw NoBodyError();
        }

        return node_to_string(messageNode->value);
    }

    void getFlags(Trees& cmdArgs, StringList& systemFlags, StringList& userFlags)
    {
        const TreeNode* flagsNode = 0;
        for (size_t idx = 1; idx < cmdArgs.size(); ++idx)
        {
            if (cmdArgs[idx].value.id().to_long() == lex_ids::FLAG_LIST)
            {
                flagsNode = &cmdArgs[idx];
                break;
            }
        }

        if (!flagsNode) return;

        for (TreeNode const& node : flagsNode->children)
        {
            string flag(node.value.begin(), node.value.end());
            if (flag.empty()) continue; // Client can send empty flag and parser accepts it
            if (flag[0] == '\\')
            {
                flag.erase(0, 1);
                systemFlags.push_back(flag);
            }
            else
            {
                userFlags.push_back(flag);
            }
        }
    }

    backend::AppendRequest makeAppendRequest()
    {
        updateStatsOnReleaseBuffer(messageBody.size());
        backend::AppendRequest request{ imapContext->userData.email,
                                        messageDateTime,
                                        systemFlags,
                                        userFlags,
                                        std::move(messageBody),
                                        DBFolderId{ destFolder.name, destFolder.fid } };
        messageBody.clear();
        return request;
    }

    Future<backend::AppendResult> append(backend::AppendRequest&& request)
    {
        return appendBackend->append(std::move(request));
    }

    void extractAppendResponse()
    {
        appendedMid = appendResult.mid;
        appendedUid = appendResult.uid;
    }

    bool messageIsDuplicate()
    {
        return appendedUid == "none";
    }

    Future<string> generateNewUid(const string& mid)
    {
        return metaBackend->regenerateImapId(mid);
    }

    void complete() override
    {
        ostringstream respCode;
        if (!appendedUid.empty())
        {
            respCode << "[APPENDUID " << destFolder.uidValidity << " " << appendedUid << "]";
        }

        completeOk(respCode.str());
    }

    string commandDump() const override
    {
        std::stringstream str;
        str << fullCommand();
        if (!appendedUid.empty())
        {
            str << " [APPENDUID " << destFolder.uidValidity << " " << appendedUid << "]"
                << " mid=" << appendedMid;
        }
        return str.str();
    }

    void updateStatsOnFillBuffer(size_t bytes)
    {
        imapContext->stats->append_buffer_size += bytes;
        imapContext->stats->append_bytes_cumulative += bytes;
    }

    void updateStatsOnReleaseBuffer(size_t bytes)
    {
        imapContext->stats->append_buffer_size -= bytes;
    }

    string destFolderName;
    string messageBody;
    string messageDateTime;
    StringList systemFlags;
    StringList userFlags;
    FolderInfo destFolder;
    backend::AppendResult appendResult;
    Future<FolderInfo> folderInfoFuture;
    string appendedUid;
    string appendedMid;
};

CommandPtr CommandAppend(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Append(commandArgs));
}
}
