#include <backend/backend.h>

#include "imap_command.h"
#include <common/settings.h>
#include <common/sequence_ranges.h>
#include <yplatform/yield.h>

namespace yimap {

struct Close : ImapSelectedCommand
{
    using YieldCtx = yplatform::yield_context<Close>;

    FolderRef mailbox;
    UidMapPtr messages;

    Close(ImapCommandArgs& cmdArgs) : ImapSelectedCommand(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mailbox = getSelectedFolder();

            unselect();

            if (canExpunge(mailbox))
            {
                yield messagesToDelete(mailbox).then(yieldCtx.capture(messages));
                if (messages->size())
                {
                    yield expunge(mailbox, messages).then(yieldCtx);
                }
            }

            completeOk();
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
    }

    FolderRef getSelectedFolder()
    {
        return imapContext->sessionState.selectedFolder;
    }

    void unselect()
    {
        imapContext->sessionState.unselect();
    }

    bool canExpunge(FolderRef mailbox)
    {
        bool isShared = imapContext->foldersCache.getSharedStatusByFid(mailbox.info().fid);
        bool isReadOnly = mailbox.readOnly();
        return !isShared && !isReadOnly;
    }

    Future<UidMapPtr> messagesToDelete(FolderRef mailbox)
    {
        seq_range range = mailbox.seqRange(false);
        range += range_t(1, static_cast<uint32_t>(mailbox.info().messageCount));
        return metaBackend->loadMessagesToDelete(mailbox, range);
    }

    Future<void> expunge(FolderRef mailbox, UidMapPtr messages)
    {
        return metaBackend->expunge(mailbox, messages);
    }

    void completeNoBackendError(const string& message)
    {
        logError() << "close error: " << message;
        completeNo("[UNAVAILABLE]", "Backend error");
    }

    UidMapPtr deletingMessages;
};

CommandPtr CommandClose(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Close(commandArgs));
}

}
