#include "message_transfer_base.h"

#include <common/settings.h>
#include <yplatform/yield.h>

namespace yimap {

struct CopyBase : MessageTransferBase
{
    using YieldCtx = yplatform::yield_context<CopyBase>;

    seq_range messageRanges;
    UidMapPtr messages;
    UidSequence srcUids;
    UidSequence dstUids;

    CopyBase(ImapCommandArgs& cmdArgs) : MessageTransferBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            selectedFolder = imapContext->sessionState.selectedFolder;
            yield loadFolderInfo(getDstFolderName()).then(yieldCtx.capture(dstFolderInfo));
            messageRanges = getMessageRanges();
            yield metaBackend->loadMessages(selectedFolder, messageRanges)
                .then(yieldCtx.capture(messages));
            if (messages->empty())
            {
                return completeEmptyMessages();
            }

            if (isSpam(selectedFolder) || isSpam(dstFolderInfo))
            {
                yield reportSpam(SpamReport::ACTION_COPY, messages).then(yieldCtx);
            }

            srcUids = messages->toUidSequence();
            yield metaBackend->copyMessages(*messages, selectedFolder.info(), dstFolderInfo)
                .then(yieldCtx.capture(dstUids));
            yield completeOkAndOutputDiff(makeCopyuidResponse(srcUids, dstUids)).then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const Utf7EncodingError&)
        {
            completeEncodingError();
        }
        catch (const NoSuchFolderError&)
        {
            completeNoSuchFolder();
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
        catch (...)
        {
            completeWithUnknownException();
        }
    }

    virtual seq_range getMessageRanges() const = 0;

    virtual string getDstFolderName() const = 0;

    virtual void completeEmptyMessages() = 0;
};

struct Copy : public CopyBase
{
    Copy(ImapCommandArgs& cmdArgs) : CopyBase(cmdArgs)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(false);
        parseSeqRange(argNode(1), res);
        return res;
    }

    string getDstFolderName() const override
    {
        return quotedArg(2);
    }

    void completeEmptyMessages() override
    {
        completeNo("[CLIENTBUG]", "Failed (no messages).");
    }
};

struct UidCopy : public CopyBase
{
    UidCopy(ImapCommandArgs& cmdArgs) : CopyBase(cmdArgs)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(true);
        parseSeqRange(argNode(1).children[0], res);
        return res;
    }

    string getDstFolderName() const override
    {
        return quoted_string(argNode(1).children[1].value);
    }

    void completeEmptyMessages() override
    {
        completeOk("[CLIENTBUG]", "Completed (no messages).");
    }
};

CommandPtr CommandCopy(ImapCommandArgs& args)
{
    return CommandPtr(new Copy(args));
}
CommandPtr CommandUidCopy(ImapCommandArgs& args)
{
    return CommandPtr(new UidCopy(args));
}
}
