#include "folder_base.h"
#include <yplatform/yield.h>

namespace yimap {

struct Create : FolderBase
{
    using YieldCtx = yplatform::yield_context<Create>;

    string mbox;
    FolderListPtr folders;
    string mboxUtf8;

    Create(ImapCommandArgs& cmdArgs) : FolderBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mbox = quotedArg(1);
            if (isInbox(mbox)) return completeBadCantApplyToInbox();
            if (inboxChildrenDenied() && isInboxChild(mbox))
                return completeBadCantApplyToInboxChild();

            yield updateFolderList().then(yieldCtx);

            folders = imapContext->foldersCache.getFolders();
            if (folders->hasFolder(mbox))
            {
                if (folders->at(mbox)->symbol != "zombie")
                {
                    return completeNoAlreadyExists();
                }
            }

            mboxUtf8 = imapContext->foldersCache.getFolders()->getOriginalName(mbox);
            yield metaBackend->createFolder(mboxUtf8).then(yieldCtx);
            yield updateFolderList().then(yieldCtx);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const FolderNameTooLargeError& e)
        {
            completeBadFolderNameToolarge(e.what());
        }
        catch (const InvalidFolderNameError& e)
        {
            completeBadFolderName();
        }
        catch (const ReservedFolderNameError& e)
        {
            completeNoReservedFolderName();
        }
        catch (const Utf7EncodingError& e)
        {
            completeBadEncodingError(mbox, e.what());
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
        catch (...)
        {
            completeNoBackendError();
        }
    }

    void completeNoAlreadyExists()
    {
        logError() << command() << " cannot create already existed folder " << mbox;
        completeNo("[CLIENTBUG]", "folder already exists.");
    }

    void completeBadCantApplyToInbox()
    {
        logError() << "cannot apply " << command() << " to INBOX";
        completeBad("[CLIENTBUG]", "cannot apply to INBOX.");
    }

    void completeBadCantApplyToInboxChild()
    {
        logError() << "cannot apply " << command() << " to INBOX subfolder";
        completeBad("[CLIENTBUG]", "cannot apply to INBOX subfolder.");
    }
};

CommandPtr CommandCreate(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Create(commandArgs));
}

}
