#include "folder_base.h"
#include <yplatform/yield.h>

namespace yimap {

struct Delete : FolderBase
{
    using YieldCtx = yplatform::yield_context<Delete>;

    string mbox;
    FolderListPtr folders;
    DBFolderId sourceFolderId;
    FullFolderInfoPtr sourceFolderInfo;

    Delete(ImapCommandArgs& cmdArgs) : FolderBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mbox = quotedArg(1);
            if (isInbox(mbox)) return completeNoCantApplyToInbox();
            if (inboxChildrenDenied() && isInboxChild(mbox))
                return completeNoCantApplyToInboxChild();

            yield updateFolderList().then(yieldCtx);
            folders = imapContext->foldersCache.getFolders();

            sourceFolderId = folders->getDBFolderId(mbox);
            sourceFolderInfo = folders->at(mbox);
            if (sourceFolderInfo->symbol == "zombie" && sourceFolderInfo->hasChildren)
            {
                return completeNoInferiorHierarchicalNames();
            }

            if (boost::iequals(settings_->allowZombie, string("no")) &&
                sourceFolderInfo->hasChildren)
            {
                return completeNoInferiorHierarchicalNames();
            }

            if (boost::iequals(settings_->allowZombie, string("log")) &&
                sourceFolderInfo->hasChildren)
            {
                // TODO remove all related code:
                // 1) no zombie folders in PG
                // 2) deleteFolder does recursive delete
                logEvent() << "DELETE: '" << sourceFolderId.name
                           << "', has inferior hierarchical names";
            }

            yield metaBackend->deleteFolder(sourceFolderId).then(yieldCtx);
            yield updateFolderList().then(yieldCtx);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const FolderNameTooLargeError& e)
        {
            completeBadFolderNameToolarge(e.what());
        }
        catch (const InvalidFolderNameError& e)
        {
            completeBadFolderName(e.what());
        }
        catch (const ReservedFolderNameError& e)
        {
            completeNoReservedFolderName(e.what());
        }
        catch (const NoSuchFolderError& e)
        {
            completeNoSuchFolder(mbox, e.what());
        }
        catch (const Utf7EncodingError& e)
        {
            completeBadEncodingError(mbox, e.what());
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
        catch (...)
        {
            completeNoBackendError();
        }
    }

    void completeNoInferiorHierarchicalNames()
    {
        logError() << command() << " '" << sourceFolderId.name
                   << "' has inferior hierarchical names";
        completeNo("", "Folder has inferior hierarchical names");
    }
};

CommandPtr CommandDelete(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Delete(commandArgs));
}

}
